<?php

namespace Drupal\better_taxonomy\Form;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Utility\Unicode;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\taxonomy\VocabularyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class ListForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Class constructor.
   *
   * @param EntityTypeManagerInterface $entityTypeManager
   *    The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): ListForm|static {
    return new static(
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Returns a unique string identifying the form.
   *
   * @return string
   *   The unique string identifying the form.
   */
  public function getFormId(): string {
    return 'better_taxonomy_list_form';
  }

  /**
   * Returns the title for the whole page.
   *
   * @param Vocabulary|null $taxonomy_vocabulary
   *   The name of the vocabulary.
   *
   * @return TranslatableMarkup
   *   The title, itself.
   */
  public function getTitle(?Vocabulary $taxonomy_vocabulary = NULL): TranslatableMarkup {
    if (!$taxonomy_vocabulary) {
      return $this->t('List');
    }
    return $this->t('%name list', ['%name' => Unicode::ucfirst($taxonomy_vocabulary->label())]);
  }

  /**
   * {@inheritDoc}
   *
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?VocabularyInterface $taxonomy_vocabulary = NULL): array {
    $form['#attached']['library'][] = 'better_taxonomy/main';
    // Clone Core taxonomy overview form.
    $this->createSectionCoreForm($form);
    // Create remove all terms.
    if (!empty($form['core']['actions'])) {
      $taxonomy_manager = $this->entityTypeManager->getStorage('taxonomy_term');
      $terms = $taxonomy_manager->loadTree($taxonomy_vocabulary->id(), 0, NULL, TRUE);
      if (!empty($terms)) {
        $options = [
          'attributes' => [
            'class' => ['action-link', 'action-link--danger', 'action-link--icon-trash'],
          ],
        ];
        $form['core']['actions']['delete_all'] = [
          '#type' => 'link',
          '#title' => $this->t('Delete all terms'),
          '#url' => Url::fromRoute('better_taxonomy.delete_all', ['taxonomy_vocabulary' => $taxonomy_vocabulary->id()], $options),
        ];
      }
    }
//    dpm($form);
    return $form;
  }

  /**
   * Appends core Taxonomy overview form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   *
   * @return void
   */
  private function createSectionCoreForm(array &$form): void {
    $route_match = \Drupal::routeMatch();
    if ($taxonomy_vocabulary = $route_match->getParameter('taxonomy_vocabulary')) {
      $form['core'] = \Drupal::formBuilder()->getForm('\Drupal\better_taxonomy\Form\CoreOverviewTerms', $taxonomy_vocabulary);
//      unset($form['core']['help']);
    }
  }

  /**
   * {@inheritDoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritDoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
  }


}
