<?php

namespace Drupal\better_taxonomy;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\taxonomy\Entity\Term;

class BTFormsService {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Class constructor.
   *
   * @param EntityTypeManagerInterface $entityTypeManager
   *    The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager) {
    $this->entityTypeManager = $entityTypeManager;
  }

  /**
   * Provides multilevel ajax selects for hierarchy selects.
   *
   * @param int $max_depth
   *   Max depth of this vocabulary.
   * @param array $tree
   *   First level terms options list.
   * @param FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *    Hierarchical selects.
   *
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function getMultilevelSelects(int $max_depth, array $tree, FormStateInterface &$form_state): array {
    $selects = [];
    $taxonomy_manager = $this->entityTypeManager->getStorage('taxonomy_term');
    for ($i = 0; $i <= $max_depth; $i++) {
      $values = $form_state->getValues();
      $options_array = [];
      if ($i == 0) {
        $options_array = [$this->t('Root')] + $tree;
      } else {
        $parent = $form_state->getValue('level_' . $i - 1);
        if (!empty($parent)) {
          $children = $taxonomy_manager->getChildren(Term::load($parent));
          $options_array = [$this->t('- Select -')];
          foreach ($children as $term) {
            $options_array[$term->id()] = $term->label();
          }
        } else {
          $form_state->setValue('level_' . $i, 0);
        }
        // With multiple levels, we must check if all previous select options are still valid.
        if (isset($values['level_' . $i]) && $values['level_' . $i] !== '0' && !in_array($values['level_' . $i], array_keys($options_array))) {
          $form_state->setValue('level_' . $i, 0);
        }
      }
      $selects['level_' . $i] = [
        '#type' => 'select',
        '#title' => $this->t('Level @level', ['@level' => $i]),
        '#level' => $i,
        '#options' => $options_array,
        '#prefix' => '<div id="level-' . $i . '">',
        '#suffix' => '</div>',
        '#states' => [
          '!visible' => [
            ':input[name="level_' . $i - 1 . '"]' => ['value' => ''],
          ],
        ],
        '#ajax' => [
          'event' => 'change',
          'callback' => ['\Drupal\better_taxonomy\Form\AddMultipleForm', 'formSelectCallback'],
          'wrapper' => 'level-' . $i + 1,
        ],
      ];
      if ($i == 0) {
        $selects['level_' . $i]['#default_value'] = 0;
        $selects['level_' . $i]['#required'] = TRUE;
      }
      // If this select has no options but the default 'select' one, hide it.
      if ($i > 0 && count($options_array) < 2) {
        $selects['level_' . $i]['#attributes']['class'] = ['hidden'];
        $selects['level_' . $i]['#title_display'] = 'invisible';
      } else {
        $selects['level_' . $i]['#attributes']['class'] = [];
        $selects['level_' . $i]['#title_display'] = 'before';
      }
    }
    return $selects;
  }

}
