<?php

namespace Drupal\better_taxonomy\Form;

use Drupal\better_taxonomy\BTFormsService;
use Drupal\better_taxonomy\BTService;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Utility\Unicode;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\taxonomy\VocabularyInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class AddMultipleForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Better taxonomy utils service.
   *
   * @var BTService
   */
  protected BTService $betterTaxonomyService;

  /**
   * The Better taxonomy forms service.
   *
   * @var BTFormsService
   */
  protected BTFormsService $betterTaxonomyFormsService;

  /**
   * Class constructor.
   *
   * @param EntityTypeManagerInterface $entityTypeManager
   *    The entity type manager.
   * @param BTService $betterTaxonomyService
   *    The Better taxonomy utils service.
   * @param BTFormsService $betterTaxonomyFormsService
   *    The Better taxonomy forms service.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager,
                              BTService $betterTaxonomyService,
                              BTFormsService $betterTaxonomyFormsService) {
    $this->entityTypeManager = $entityTypeManager;
    $this->betterTaxonomyService = $betterTaxonomyService;
    $this->betterTaxonomyFormsService = $betterTaxonomyFormsService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): ListForm|static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('better_taxonomy.better_taxonomy_service'),
      $container->get('better_taxonomy.better_taxonomy_forms_service'),
    );
  }

  /**
   * Returns a unique string identifying the form.
   *
   * @return string
   *   The unique string identifying the form.
   */
  public function getFormId(): string {
    return 'better_taxonomy_add_multiple_form';
  }

  /**
   * Returns the title for the whole page.
   *
   * @param Vocabulary $taxonomy_vocabulary
   *   The name of the vocabulary.
   *
   * @return TranslatableMarkup
   *   The title, itself.
   */
  public function getTitle(Vocabulary $taxonomy_vocabulary): TranslatableMarkup {
    return $this->t('Add multiple terms in %name', ['%name' => Unicode::ucfirst($taxonomy_vocabulary->label())]);
  }

  /**
   * Form constructor.
   *
   * Display a tree of all the terms in a vocabulary, with options to edit
   * each one. The form is made drag and drop by the theme function.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param FormStateInterface $form_state
   *   The current state of the form.
   * @param VocabularyInterface|null $taxonomy_vocabulary
   *   The vocabulary to display the overview form for.
   *
   * @return array
   *   The form structure.
   *
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?VocabularyInterface $taxonomy_vocabulary = NULL): array {
    $form['#attached']['library'][] = 'better_taxonomy/main';
    // Send vocabulary to the form submission.
    $form['vocabulary'] = [
      '#type' => 'hidden',
      '#value' => $taxonomy_vocabulary->id(),
    ];

    // Retrieve vocabulary max depth.
    $taxonomy_manager = $this->entityTypeManager->getStorage('taxonomy_term');
    $vocabulary_tree = $taxonomy_manager->loadTree($taxonomy_vocabulary->label(), 0, NULL, TRUE);
    $max_depth = $this->betterTaxonomyService->getVocabularyMaxDepth($taxonomy_vocabulary->id(), $vocabulary_tree);
    if ($max_depth > 0) {
      $form_state->set('max_depth', $max_depth);
      // Get Taxonomy tree & multilevel select.
      $form['selects'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Select the parent term where the new ones should be added.'),
      ];
      // Builds first level terms options list.
      $tree = [];
      foreach ($vocabulary_tree as $term) {
        if ($term->depth == 0) {
          $tree[$term->id()] = $term->label();
        }
      }
      $multilevel_selects = $this->betterTaxonomyFormsService->getMultilevelSelects($max_depth, $tree, $form_state);
      foreach ($multilevel_selects as $multilevel_select) {
        $form['selects']['level_' . $multilevel_select['#level']] = $multilevel_select;
      }
    }

    $instructions = [];
    $instructions[] = $this->t('One term per line.');
    $instructions[] = $this->t('Hierarchy of terms to be added must be indicated with a single dash "-" per level.');
    $instructions[] = $this->t('Example:');
    $instructions[] = $this->t('Plants');
    $instructions[] = $this->t('-Trees');
    $instructions[] = $this->t('--Oak');
    $instructions[] = $this->t('Flowers');
    $instructions[] = $this->t('-Rose');
    $form['terms'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Terms'),
      '#description' => implode('<br/>', $instructions),
      '#rows' => 8,
      '#required' => TRUE,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add terms'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'onclick' => 'javascript:var s=this;setTimeout(function(){s.value="Saving...";s.disabled=true;},1);',
      ],
    ];
    return $form;
  }

  /**
   * Rebuild parent tree of select on ajax callback.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return AjaxResponse
   *    Rebuilds de selects.
   */
  public static function formSelectCallback(array $form, FormStateInterface $form_state): AjaxResponse {
    $max_depth = $form_state->get('max_depth');
    $response = new AjaxResponse();
    for ($i = 1; $i <= $max_depth; $i++) {
      $response->addCommand(new ReplaceCommand('#level-' . $i, $form['selects']['level_' . $i]));
    }
    return $response;
  }

  /**
   * {@inheritDoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
  }

  /**
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return void
   *
   * @throws EntityStorageException
   * @throws EntityMalformedException
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $values = $form_state->getValues();
    // Save terms.
    $saved_terms = $this->betterTaxonomyService->addMultipleTerms($values);
    if (count($saved_terms)) {
      $message = $this->t('@count term(s) added.', ['@count' => count($saved_terms)]);
    }
    else {
      $message = $this->t('There were no terms to add.');
    }
    $this->messenger()->addMessage($message);
    $vid = $values['vocabulary'];
    $taxonomy_vocabulary = Vocabulary::load($vid);
    $redirect_url = $taxonomy_vocabulary->toUrl('overview-form');
    $form_state->setRedirectUrl($redirect_url);
  }
}
