<?php

namespace Drupal\Tests\bibcite_footnotes\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\filter\FilterProcessResult;
use Drupal\Tests\bibcite_entity\Traits\EntityCreationTrait;
use Drupal\Core\Render\RendererInterface;

/**
 * Kernel tests for ReferenceFootnotesFilter with real entities.
 *
 * @group bibcite_footnotes
 */
class ReferenceFootnotesFilterKernelTest extends KernelTestBase
{

  use EntityCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'field',
    'text',
    'user',
    'bibcite',
    'bibcite_entity',
    'bibcite_footnotes',
    'serialization',
  ];

  /**
   * The filter plugin.
   *
   * @var \Drupal\bibcite_footnotes\Plugin\Filter\ReferenceFootnotesFilter
   */
  protected $filter;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Test reference entities.
   *
   * @var \Drupal\bibcite_entity\Entity\ReferenceInterface[]
   */
  protected $references = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('bibcite_reference');
    $this->installEntitySchema('bibcite_contributor');
    $this->installEntitySchema('bibcite_keyword');
    $this->installConfig(['bibcite', 'bibcite_entity', 'system']);

    // Create test reference entities.
    $this->references[] = $this->createReference([
      'title' => 'Anne of Green Gables',
      'bibcite_year' => 1908,
      'type' => 'book',
    ]);

    $this->references[] = $this->createReference([
      'title' => 'The Fort Bragg Cartel',
      'bibcite_year' => 2025,
      'type' => 'book',
    ]);

    // Get the filter plugin and renderer.
    $this->filter = $this->container->get('plugin.manager.filter')
      ->createInstance('filter_reference_footnotes');
    $this->renderer = $this->container->get('renderer');
  }

  /**
   * Tests filter processing with real reference entities.
   */
  public function testFilterWithRealReferences()
  {
    $reference = $this->references[0];

    $text = '<p>Test <bibcite-footnote data-entity-id="' . $reference->id() . '" data-page-range="12"></bibcite-footnote> citation.</p>';

    // Wrap the filter processing in a render context
    $result = $this->renderer->executeInRenderContext(new \Drupal\Core\Render\RenderContext(), function () use ($text) {
      return $this->filter->process($text, 'en');
    });

    $this->assertInstanceOf(FilterProcessResult::class, $result);
    $this->assertStringContainsString('bibcite-citation', $result->getProcessedText());
    $this->assertStringNotContainsString('<bibcite-footnote', $result->getProcessedText());

    // Should contain the bibliography section
    $this->assertStringContainsString('bibcite-footnotes-section', $result->getProcessedText());
    $this->assertStringContainsString('Bibliography', $result->getProcessedText());
  }

  /**
   * Tests multiple references in same text.
   */
  public function testMultipleReferences()
  {
    $reference1 = $this->references[0];
    $reference2 = $this->references[1];

    $text = '<p>First <bibcite-footnote data-entity-id="' . $reference1->id() . '" data-page-range="12"></bibcite-footnote> and second <bibcite-footnote data-entity-id="' . $reference2->id() . '" data-page-range="22"></bibcite-footnote> citation.</p>';

    // Wrap the filter processing in a render context
    $result = $this->renderer->executeInRenderContext(new \Drupal\Core\Render\RenderContext(), function () use ($text) {
      return $this->filter->process($text, 'en');
    });

    $this->assertInstanceOf(FilterProcessResult::class, $result);

    // Should have two citations
    $this->assertEquals(2, substr_count($result->getProcessedText(), 'bibcite-citation'));

    // Should have bibliography section
    $this->assertStringContainsString('bibcite-footnotes-section', $result->getProcessedText());
  }

  /**
   * Tests same reference used multiple times.
   */
  public function testSameReferenceMultipleTimes()
  {
    $reference = $this->references[0];

    $text = '<p>First <bibcite-footnote data-entity-id="' . $reference->id() . '" data-page-range="12"></bibcite-footnote> and second <bibcite-footnote data-entity-id="' . $reference->id() . '" data-page-range="34"></bibcite-footnote> citation.</p>';

    // Wrap the filter processing in a render context
    $result = $this->renderer->executeInRenderContext(new \Drupal\Core\Render\RenderContext(), function () use ($text) {
      return $this->filter->process($text, 'en');
    });

    $this->assertInstanceOf(FilterProcessResult::class, $result);

    // Should have two citations
    $this->assertEquals(2, substr_count($result->getProcessedText(), 'bibcite-citation'));
  }

  /**
   * Tests footnote without entity ID.
   */
  public function testFootnoteWithoutEntityId()
  {
    $text = '<p>Test <bibcite-footnote data-page-range="12"></bibcite-footnote> invalid citation.</p>';

    // Wrap the filter processing in a render context
    $result = $this->renderer->executeInRenderContext(new \Drupal\Core\Render\RenderContext(), function () use ($text) {
      return $this->filter->process($text, 'en');
    });

    $this->assertInstanceOf(FilterProcessResult::class, $result);

    // Should not contain invalid footnotes
    $this->assertStringNotContainsString('bibcite-footnote', $result->getProcessedText());
    $this->assertStringNotContainsString('bibcite-citation', $result->getProcessedText());
  }

  /**
   * Tests text without any footnotes.
   */
  public function testTextWithoutFootnotes()
  {
    $text = '<p>This is a simple paragraph without any footnotes.</p>';

    // Wrap the filter processing in a render context
    $result = $this->renderer->executeInRenderContext(new \Drupal\Core\Render\RenderContext(), function () use ($text) {
      return $this->filter->process($text, 'en');
    });

    $this->assertInstanceOf(FilterProcessResult::class, $result);
    $this->assertEquals($text, $result->getProcessedText());
  }

  /**
   * Tests that the filter returns original text when no footnotes are present.
   */
  public function testNoFootnotesInText()
  {
    $text = '<p>This text has no footnotes at all.</p>';

    // Wrap the filter processing in a render context
    $result = $this->renderer->executeInRenderContext(new \Drupal\Core\Render\RenderContext(), function () use ($text) {
      return $this->filter->process($text, 'en');
    });

    $this->assertInstanceOf(FilterProcessResult::class, $result);
    $this->assertEquals($text, $result->getProcessedText());
  }
}
