const path = require('path');
const fs = require('fs');
const webpack = require('webpack');
const { styles, builds } = require('@ckeditor/ckeditor5-dev-utils');
const TerserPlugin = require('terser-webpack-plugin');

function getDirectories(srcpath) {
  return fs
    .readdirSync(srcpath)
    .filter((item) => fs.statSync(path.join(srcpath, item)).isDirectory());
}

module.exports = (env, argv) => {
  const isProduction = argv.mode === 'production';

  const configs = [];

  // Loop through every subdirectory in src, each a different plugin, and build
  // each one in ./build.
  getDirectories('./js/ckeditor5_plugins').forEach((dir) => {
    const bc = {
      mode: argv.mode || 'development',
      devtool: isProduction ? false : 'source-map',
      optimization: {
        minimize: isProduction,
        minimizer: isProduction ? [
          new TerserPlugin({
            terserOptions: {
              ecma: 2020, // or a higher version
              parse: {
                ecma: 2020,
              },
              compress: {
                ecma: 5,
                defaults: true,
              },
              format: {
                comments: false,
              },
            },
            test: /\.js(\?.*)?$/i,
            extractComments: false,
          }),
        ] : [],
        moduleIds: 'named',
      },
      entry: {
        path: path.resolve(
          __dirname,
          'js/ckeditor5_plugins',
          dir,
          'src/index.js',
        ),
      },
      output: {
        path: path.resolve(__dirname, isProduction ? './js/build' : './js'),
        filename: `${dir}.js`,
        library: ['CKEditor5', dir],
        libraryTarget: 'umd',
        libraryExport: 'default',
      },
      resolve: {
        alias: {
          core: path.join(__dirname, 'core'),
        },
      },
      plugins: [
        new webpack.optimize.LimitChunkCountPlugin({
          maxChunks: 1,
        }),
        // It is possible to require the ckeditor5-dll.manifest.json used in
        // core/node_modules rather than having to install CKEditor 5 here.
        // However, that requires knowing the location of that file relative to
        // where your module code is located.
        new webpack.DllReferencePlugin({
          manifest: require('ckeditor5/build/ckeditor5-dll.manifest.json'), // eslint-disable-line global-require, import/no-unresolved
          scope: 'ckeditor5/src',
          name: 'CKEditor5.dll',
        }),
      ],
      module: {
        rules: [{ test: /\.svg$/, use: 'raw-loader' }],
      },
    };

    configs.push(bc);
  });

  return configs;
};
