<?php

namespace Drupal\bibliocommons;

use Drupal\Core\Config\ConfigFactory;
use Drupal\key\KeyRepository;
use Drupal\wsdata\WSDataService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Language\LanguageManagerInterface;

/**
 * Fetch and format books.
 */
class BibliocommonsService {
  /**
   * Config Factory for Drupal.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  protected ConfigFactory $configFactory;

  /**
   * Key repository.
   *
   * @var \Drupal\key\KeyRepository
   */
  protected KeyRepository $keyRepository;

  /**
   * Service for WSData used in caching.
   *
   * @var \Drupal\wsdata\WSDataService
   */
  protected WSDataService $wsdata;

  /**
   * Language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Constructor to create object.
   *
   * @param \Drupal\Core\Config\ConfigFactory $configFactory
   *   Config factory to get bibliocommons config.
   * @param \Drupal\key\KeyRepository $keyRepository
   *   Key repository service.
   * @param \Drupal\wsdata\WSDataService $wsdata
   *   Service used for caching.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *  Language manager service.
   */
  public function __construct(ConfigFactory $configFactory, KeyRepository $keyRepository, WSDataService $wsdata, LanguageManagerInterface $languageManager) {
    $this->configFactory = $configFactory;
    $this->keyRepository = $keyRepository;
    $this->wsdata = $wsdata;
    $this->languageManager = $languageManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('http_client'),
      $container->get('config.factory'),
      $container->get('wsdata'),
      $container->get('language_manager')
    );
  }

  /**
   * Returns book list from bibliocommons.
   *
   * @return array
   *   Render array.
   */
  public function getBookList($id, $limit = 6, $type = 'lists') {
    // Get the config for a book list request.
    $config = $this->configFactory->get('bibliocommons.settings');
    $keyId = $config->get('api_key');
    $key = trim($this->keyRepository->getKey($keyId)->getKeyValue());
    $library = trim($config->get('library_id'));
    $locale = $this->languageManager->getCurrentLanguage()->getId();

    $data = [
      'api_key' => $key,
      'library' => $library,
      'locale' => $locale,
    ];

    if ($type === 'lists') {
      $books = $this->wsdata->call('books', 'get', ['ID' => $id], $data);
    }
    else {
      $books = $this->wsdata->call('book', 'get', ['ID' => $id], $data);
    }

    // If the book request fails, an empty array will be returned.
    if ($books && $limit > 0) {
      // Remove all books except for the number of books limited.
      // This needs to be done as it is not clear if there is a way to
      // limit the books on the request.
      if ($type === 'lists') {
        $books['list']['list_items'] = array_slice($books['list']['list_items'], 0, $limit);
      }
    }
    elseif ($books && $limit == 0) {
      // Do nothing if we have books and limit is zero.
    }
    else {
      $books = [];
    }

    return $books;
  }

}
