<?php

namespace Drupal\bibliocommons\Plugin\Field\FieldFormatter;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\bibliocommons\BibliocommonsService;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Bibliocommons formatters.
 */
trait BibliocommonsFormatterTrait {

  /**
   * Constructs a BibliocommonsService instance.
   *
   * @param \Drupal\bibliocommons\BibliocommonsService $booksService
   *   The books service to integrate with bilbiocommons.
   */
  protected BibliocommonsService $booksService;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a module handler instance.
   *
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   Module handler.
   */
  protected ModuleHandler $moduleHandler;

  /**
   * Store a logger to print error messages.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger.
   */
  protected LoggerInterface $logger;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'show_image' => 1,
      'use_large_cover_img' => 0,
      'show_title' => 1,
      'show_author' => 0,
      'show_subtitle' => 0,
      'show_publication_date' => 0,
      'show_my_shelf_link' => 0,
      'show_hold_link' => 0,
      'show_description' => 0,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary[] = $this->t('Show title: @show', ['@show' => $this->getSetting('show_title')]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements['show_image'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Image'),
      '#default_value' => $this->getSetting('show_image'),
      '#description' => $this->t('Display book Image.'),
    ];

    $elements['use_large_cover_img'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use large cover image'),
      '#default_value' => $this->getSetting('use_large_cover_img'),
      '#description' => $this->t('Use the large cover image instead of the default (medium).'),
    ];

    $elements['show_title'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Title'),
      '#default_value' => $this->getSetting('show_title'),
      '#description' => $this->t('Display book title.'),
    ];

    $elements['show_author'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Author Name'),
      '#default_value' => $this->getSetting('show_author'),
      '#description' => $this->t('Display author name.'),
    ];

    $elements['show_subtitle'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Subtitle'),
      '#default_value' => $this->getSetting('show_subtitle'),
      '#description' => $this->t('Display book subtitle.'),
    ];

    $elements['show_publication_date'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Publication Date'),
      '#default_value' => $this->getSetting('show_publication_date'),
      '#description' => $this->t('Display book publication date.'),
    ];

    $elements['show_my_shelf_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show My Shelf link'),
      '#default_value' => $this->getSetting('show_my_shelf_link'),
      '#description' => $this->t('Display add to my shelf link.'),
    ];

    $elements['show_hold_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Place Hold link'),
      '#default_value' => $this->getSetting('show_hold_link'),
      '#description' => $this->t('Display place a hold link.'),
    ];

    $elements['show_description'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Description'),
      '#default_value' => $this->getSetting('show_description'),
      '#description' => $this->t('Display book description.'),
    ];
    return $elements;
  }

  /**
   * Constructs and returns a syndetics book cover URL.
   *
   * @param bool $lg_img
   *   Indicates whether or not the larger image is preferred.
   * @param string $image_id
   *   The image ID.
   * @param string $image_upc
   *   The image UPC.
   * @param string $client_id
   *   The syndetics client ID.
   *
   * @return string
   *   Returns a syndetics URL for book cover image.
   */
  protected function getSyndeticsCoverUrl($lg_img, $image_id, $image_upc, $client_id) {
    $image_base_url = 'https://secure.syndetics.com/index.aspx?isbn=';
    $image_size_comp = $lg_img ? 'LC' : 'MC';
    return isset($image_id) || isset($image_upc) ? $image_base_url . $image_id . '/' . $image_size_comp . '.GIF&client=' . $client_id . '&upc=' . $image_upc : '';
  }

  /**
   * Gets a valid URL to an image to use as book cover.
   *
   * @param bool $lg_img
   *   Indicates whether or not the larger image is preferred.
   * @param string $image_id
   *   The image ID.
   * @param string $image_upc
   *   The image UPC.
   * @param string $client_id
   *   The syndetics client ID.
   * @param string $book_class
   *   The bibliocommons book class.
   * @param string $jacket_cover_url
   *   The jacket cover URL returned from bibliocommons.
   * @param bool $use_jacket_cover
   *   Whether or not the jacket cover URL should be used as preferred image.
   *
   * @return string
   *   A valid URL to an image to use as book cover.
   */
  protected function getBookCoverUrl($lg_img, $image_id, $image_upc, $client_id, $book_class, $jacket_cover_url = '', $use_jacket_cover = FALSE) {
    // Check if we should be using the syndetics image, or if the jacket
    // cover URL should be used.
    $cover_url = '';
    if (!empty($jacket_cover_url) and $use_jacket_cover) {
      // Use the jacket cover URL for the image.
      $cover_url = $jacket_cover_url;
    }
    else {
      // Construct syndetics image URL.
      $cover_url = $this->getSyndeticsCoverUrl($lg_img, $image_id, $image_upc, $client_id, $book_class);
    }

    // If the cover url is not set then use a backup image.
    // When the image doesn't exist but we have a valid book we
    // get a single pixel. So check the size to determine if the image
    // is valid.
    $min_image_width = 5;
    try {
      // If an error is thrown with get image size
      // then set the image to the default.
      $image_size = getimagesize($cover_url);
    }
    catch (\ValueError $e) {
      $this->logger->error('Failed to get image size of the book cover: ' . $e->getMessage());
      $image_size = FALSE;
    }
    // Fall-back to the medium size cover if the option to show large
    // cover images is on and there is no image.
    if (
      $image_size === FALSE ||
      empty($cover_url) |
      ($image_size && $image_size[0] < $min_image_width)) {
      if ($use_jacket_cover) {
        // Default is to use jacket cover URL, but that didn't give us
        // a valid image. Fall back to using syndetics image.
        return $this->getBookCoverUrl($lg_img, $image_id, $image_upc, $client_id, $book_class);
      }
      elseif ($lg_img) {
        // Attempted to get large syndetics image, fall back to medium.
        return $this->getBookCoverUrl(FALSE, $image_id, $image_upc, $client_id, $book_class);
      }
      // Last resort: use default image.
      $module_path = $this->moduleHandler->getModule('bibliocommons')->getPath();
      $cover_url = '/' . $module_path . '/assets/images/icon-' . strtolower($book_class) . '.png';
    }
    return $cover_url;
  }

  /**
   * Constructs a "hold" link for a book.
   *
   * @param string $book_id
   *   The ID of a book.
   * @param string $library_id
   *   The library ID.
   */
  protected function getBookHoldLink($book_id, $library_id) {
    return 'https://' . $library_id . '.bibliocommons.com/holds/select_hold/' . $book_id;
  }

  /**
   * Constructs a "add to my shelf" link for a book.
   *
   * @param string $book_id
   *   The ID of a book.
   * @param string $library_id
   *   The library ID.
   */
  protected function getBookMyShelfLink($book_id, $library_id) {
    return 'https://' . $library_id . '.bibliocommons.com/collection/add/my/library?bib=' . $book_id . '&bib_status=past';
  }

}
