<?php

namespace Drupal\bibliocommons\Plugin\Field\FieldFormatter;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\bibliocommons\BibliocommonsService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'Book' formatter.
 *
 * @FieldFormatter(
 *   id = "bibliocommons_book",
 *   label = @Translation("Book formatter"),
 *   field_types = {
 *     "string"
 *   }
 * )
 */
class BookFormatter extends FormatterBase {
  use BibliocommonsFormatterTrait;

  /**
   * Define type for API call and template to use.
   *
   * @var string
   */
  const TYPE = 'titles';

  /**
   * Constructs a BookListFormatter instance.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings settings.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\bibliocommons\BibliocommonsService $booksService
   *   The books service to integrate with bilbiocommons.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   The module handler to get module related information.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    ConfigFactoryInterface $configFactory,
    BibliocommonsService $booksService,
    ModuleHandler $moduleHandler,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->configFactory = $configFactory;
    $this->booksService = $booksService;
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('config.factory'),
      $container->get('bibliocommons.books_service'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $result = [];
    $fields = $items->getValue();

    foreach ($fields as $field) {
      // Continue if the entity doesn't have the required fields.
      $value = $field["value"];

      if ($value) {
        $result[] = $this->buildBook($value);
      }
    }

    return $result;
  }

  /**
   * Constructs a book array.
   *
   * @param int $book_id
   *   The book id to grab the book information.
   *
   * @return array
   *   The render array for books.
   */
  private function buildBook($book_id) {
    $config = $this->configFactory->get('bibliocommons.settings');

    // Set up the render array.
    $result = [
      '#theme' => 'bibliocommons_books',
      '#books' => [],
      '#title' => '',
      '#my_shelf_link' => '',
      '#hold_link' => '',
      '#cover_url' => '',
      '#details_url' => '',
      '#type' => 'titles',
      '#attached' => [
        'library' => ['bibliocommons/bibliocommons'],
      ],
    ];

    // Get book by ID with request.
    $book_id = intval($book_id);
    $books = $this->booksService->getBookList($book_id, 1, self::TYPE);

    // Handle the book if we have one.
    if (count($books) > 0) {
      $book = $books['title'];

      // Add the book title and url.
      if ($this->getSetting('show_title')) {
        $result['#title'] = $book['title'];
        $result['#details_url'] = $book["details_url"];
      }
      // Add optional image cover.
      if ($this->getSetting('show_image')) {
        // Set the cover URL.
        // For French books, attempt to use jacket cover URL.
        $jacket_cover_url = $book['jacket_cover']['url'];
        $use_jacket_cover = (
          in_array('French', array_column($book['languages'], 'name'))
          and !empty($jacket_cover_url)
        );
        $cover_url = $this->getBookCoverURL(
          $this->getSetting('use_large_cover_img'),
          $book['isbns'][0] ?? NULL,
          $book['upcs'][0] ?? NULL,
          $config->get('client_id'),
          $book['format']['id'],
          $book['jacket_cover']['url'],
          $use_jacket_cover
        );

        $result['#cover_url'] = $cover_url;
      }
      // Add optional info about the book.
      if ($this->getSetting('show_author')) {
        $result['#books'][0]['author'] = $book['authors'][0]['name'];
      }
      if ($this->getSetting('show_subtitle')) {
        $result['#books'][0]['subtitle'] = $book['sub_title'];
      }
      if ($this->getSetting('show_publication_date')) {
        $result['#books'][0]['publication_date'] = $book['publication_date'];
      }
      if ($this->getSetting('show_my_shelf_link')) {
        $result['#my_shelf_link'] = $this->getBookMyShelfLink($book_id, $config->get('library_id'));
      }
      if ($this->getSetting('show_hold_link')) {
        $result['#hold_link'] = $this->getBookHoldLink($book_id, $config->get('library_id'));

      }
      if ($this->getSetting('show_description')) {
        $result['#books'][0]['description'] = $book['description'];
      }
      return $result;
    }
  }

}
