<?php

namespace Drupal\bibliocommons\Plugin\Field\FieldFormatter;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\bibliocommons\BibliocommonsService;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'Booklist' formatter.
 *
 * @FieldFormatter(
 *   id = "bibliocommons_booklist_collection",
 *   label = @Translation("Book List Collection formatter"),
 *   field_types = {
 *     "string"
 *   }
 * )
 */
class BookListCollectionFormatter extends FormatterBase {
  use BibliocommonsFormatterTrait;

  /**
   * Define type for API call and template to use.
   *
   * @var string
   */
  const TYPE = 'lists';

  /**
   * Constructs a BookListCollectionFormatter instance.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings settings.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\bibliocommons\BibliocommonsService $booksService
   *   The books service to integrate with bilbiocommons.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   The module handler to get module related information.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger to log error messages.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    ConfigFactoryInterface $configFactory,
    BibliocommonsService $booksService,
    ModuleHandler $moduleHandler,
    LoggerInterface $logger,
  ) {
    parent::__construct(
      $plugin_id, $plugin_definition, $field_definition,
      $settings, $label, $view_mode, $third_party_settings,
      $logger);
    $this->booksService = $booksService;
    $this->configFactory = $configFactory;
    $this->moduleHandler = $moduleHandler;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('config.factory'),
      $container->get('bibliocommons.books_service'),
      $container->get('module_handler'),
      $container->get('logger.factory')->get('bibliocommons'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $result = [];
    $fields = $items->getValue();

    foreach ($fields as $field) {
      // Continue if the entity doesn't have the required fields.
      $value = $field["value"];

      if ($value) {
        $parent = $items->getParent();
        $result = $this->buildBook($value, $parent->getEntity());
        break;
      }
    }

    return $result;
  }

  /**
   * Constructs a book array.
   *
   * @param int $book_id
   *   The entity to grab the book information.
   * @param bool $parent_entity
   *   The parent entity the field formatter is on.
   *
   * @return array
   *   The array for book and book list.
   */
  private function buildBook($book_id, $parent_entity) {
    $book = [];
    $cover_url = '';
    $books_display = [];
    $details_url = '';
    $config = $this->configFactory->get('bibliocommons.settings');

    // Get book list by ID with request.
    $book_id = intval($book_id);
    $limit = $this->getSetting('limit');
    $books = $this->booksService->getBookList($book_id, $limit, self::TYPE);
    $show_more_link = filter_var($this->getSetting('show_more_link'),
      FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
    $show_author_info = filter_var($this->getSetting('show_author_info'),
      FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);

    // Structure books into a data structure.
    if (count($books) > 0) {
      // Extract book list from structure.
      $title = '';
      $format = '';
      $jacket_cover_url = '';
      $book_id = '';
      $cover_url = '';
      $details_url = '';
      $list_description = $books["list"]["description"];
      $list_author_info = $books["list"]["user"];
      $list_name = $books["list"]["name"];
      $list_details_url = $books["list"]["details_url"];
      $books = $books['list']['list_items'];
      $module_path = $this->moduleHandler->getModule('bibliocommons')->getPath();

      foreach ($books as $book) {
        if ($book["list_item_type"] === 'url') {
          // Handle case where we have a url list item type.
          // This has a different structure then a list item type of title.
          $url = $book['url'];
          $title = $url['title'];
          $details_url = $url['url'];

          // Set the cover image since there is none provided from the API.
          if ($this->getSetting('show_image')) {
            $cover_url = '/' . $module_path . '/assets/images/icon-bk.png';
          }
        }
        else {
          $book = $book['title'];
          $book_id = $book['id'];
          $title = $book['title'];
          $format = $book['format']['name'];
          $jacket_cover_url = $book['jacket_cover']['url'];
          $details_url = $book['details_url'];
          // For French books, attempt to use jacket cover URL.
          $use_jacket_cover = (
            in_array('French', array_column($book['languages'], 'name'))
            and !empty($jacket_cover_url)
          );
          if ($this->getSetting('show_image')) {
            $cover_url = $this->getBookCoverURL(
              $this->getSetting('use_large_cover_img'),
              $book['isbns'][0] ?? NULL,
              $book['upcs'][0] ?? NULL,
              $config->get('client_id'),
              $book['format']['id'],
              $book['jacket_cover']['url'],
              $use_jacket_cover
            );
          }
        }

        $temp_book = [
          'title' => $title,
          'cover_url' => $cover_url,
          'details_url' => $details_url,
          'format' => $format,
          'jacket_cover_url' => $jacket_cover_url,
          'book_id' => $book_id,
        ];
        // Add optional info about the book.
        if ($this->getSetting('show_author')) {
          $temp_book['author'] = $book['authors'][0]['name'];
        }
        if ($this->getSetting('show_subtitle')) {
          $temp_book['subtitle'] = $book['sub_title'];
        }
        if ($this->getSetting('show_publication_date')) {
          $temp_book['publication_date'] = $book['publication_date'];
        }
        if ($this->getSetting('show_my_shelf_link')) {
          $temp_book['my_shelf_link'] = $this->getBookMyShelfLink($book_id, $config->get('library_id'));
        }
        if ($this->getSetting('show_hold_link')) {
          $temp_book['hold_link'] = $this->getBookHoldLink($book_id, $config->get('library_id'));
        }
        if ($this->getSetting('show_description')) {
          $temp_book['description'] = $book['description'];
        }

        array_push($books_display, $temp_book);
      }
    }

    $result = [
      '#theme' => 'bibliocommons_books_collection',
      '#parent_entity' => $parent_entity,
      '#books' => $books_display,
      '#title' => $list_name,
      '#list_description' => $list_description,
      '#author_link' => $list_author_info["profile_url"],
      '#author_name' => $list_author_info["name"],
      '#show_more_link' => $show_more_link,
      '#show_author_info' => $show_author_info,
      '#cover_url' => '',
      '#details_url' => $list_details_url,
      '#type' => self::TYPE,
      '#attached' => [
        'library' => ['bibliocommons/bibliocommons'],
      ],
    ];

    return $result;
  }

}
