<?php

namespace Drupal\bibliocommons\Plugin\Field\FieldFormatter;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\bibliocommons\BibliocommonsService;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'Booklist' formatter.
 *
 * @FieldFormatter(
 *   id = "bibliocommons_booklist",
 *   label = @Translation("Book List formatter"),
 *   field_types = {
 *     "string"
 *   }
 * )
 */
class BookListFormatter extends FormatterBase {
  /**
   * Define type for API call and template to use.
   *
   * @var string
   */
  const TYPE = 'lists';

  /**
   * Constructs a BibliocommonsService instance.
   *
   * @param \Drupal\bibliocommons\BibliocommonsService $booksService
   *   The books service to integrate with bilbiocommons.
   */
  protected BibliocommonsService $booksService;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a module handler instance.
   *
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   Module handler.
   */
  protected ModuleHandler $moduleHandler;

  /**
   * Store a logger to print error messages.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   Logger.
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a BookListFormatter instance.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings settings.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\bibliocommons\BibliocommonsService $booksService
   *   The books service to integrate with bilbiocommons.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *   The module handler to get module related information.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger to log error messages.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    ConfigFactoryInterface $configFactory,
    BibliocommonsService $booksService,
    ModuleHandler $moduleHandler,
    LoggerInterface $logger,
  ) {
    parent::__construct(
      $plugin_id, $plugin_definition, $field_definition,
      $settings, $label, $view_mode, $third_party_settings,
      $logger);
    $this->booksService = $booksService;
    $this->configFactory = $configFactory;
    $this->moduleHandler = $moduleHandler;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('config.factory'),
      $container->get('bibliocommons.books_service'),
      $container->get('module_handler'),
      $container->get('logger.factory')->get('bibliocommons'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'limit' => 0,
      'show_more_link' => 1,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary[] = $this->t('Limit: @limit', ['@limit' => $this->getSetting('limit')]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {

    $elements['limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Limit'),
      '#default_value' => $this->getSetting('limit'),
      '#description' => $this->t('Set the limit for the number of books to display. Setting 0 will display all books.'),
      '#min' => 0,
    ];

    $elements['show_more_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show More Link'),
      '#default_value' => $this->getSetting('show_more_link'),
      '#description' => $this->t('Show more link will display a link to the book list.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $result = [];
    $fields = $items->getValue();

    foreach ($fields as $field) {
      // Continue if the entity doesn't have the required fields.
      $value = $field["value"];

      if ($value) {
        $parent = $items->getParent();
        $result = $this->buildBook($value, $parent->getEntity());
        break;
      }
    }

    return $result;
  }

  /**
   * Constructs a book array.
   *
   * @param int $book_id
   *   The entity to grab the book information.
   * @param bool $parent_entity
   *   The parent entity the field formatter is on.
   *
   * @return array
   *   The array for book and book list.
   */
  private function buildBook($book_id, $parent_entity) {
    $book = [];
    $image_id = '';
    $cover_url = '';
    $image_base_url = 'https://secure.syndetics.com/index.aspx?isbn=';
    $book_class = '';
    $books_display = [];
    $details_url = '';
    $config = $this->configFactory->get('bibliocommons');

    // Get book list by ID with request.
    $book_id = intval($book_id);
    $limit = $this->getSetting('limit');
    $books = $this->booksService->getBookList($book_id, $limit, self::TYPE);
    $show_more_link = filter_var($this->getSetting('show_more_link'),
      FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);

    // Structure books into a data structure.
    if (count($books) > 0) {
      // Extract book list from structure.
      $title = '';
      $cover_url = '';
      $details_url = '';
      $list_name = $books["list"]["name"];
      $list_details_url = $books["list"]["details_url"];
      $books = $books['list']['list_items'];
      $module_path = $this->moduleHandler->getModule('bibliocommons')->getPath();
      $min_image_width = 5;

      foreach ($books as $book) {
        if ($book["list_item_type"] === 'url') {
          // Handle case where we have a url list item type.
          // This has a different structure then a list item type of title.
          $url = $book['url'];
          $title = $url['title'];
          $details_url = $url['url'];

          // Set the cover image since there is none provided from the API.
          $book_class = strtolower($book['format']['id']);
          $cover_url = $module_path . '/assets/images/icon-bk.png';
        }
        else {
          $book = $book['title'];
          $image_id = $book['isbns'][0] ?? NULL;
          $image_upc = $book['upcs'][0] ?? NULL;
          $cover_url = isset($image_id) || isset($image_upc) ? $image_base_url . $image_id . '/MC.GIF&client=' . $config->get('client_id') . '&upc=' . $image_upc : '';
          $title = $book['title'];
          $details_url = $book['details_url'];

          // If the cover url is not set then use a backup image.
          // When the image doesn't exist but we have a valid book we
          // get a single pixel. So check the size to determine if the image
          // is valid.
          try {
            // If an error is thrown with get image size
            // then set the image to the default.
            $image_size = getimagesize($cover_url);
          }
          catch (\ValueError $e) {
            $this->logger->error('Failed to get image size of the book cover: ' . $e->getMessage());
            $image_size = FALSE;
          }

          if (
            $image_size === FALSE ||
            empty($cover_url) ||
            ($image_size && $image_size[0] < $min_image_width)) {
            $book_class = strtolower($book['format']['id']);
            $cover_url = '/' . $module_path . '/assets/images/icon-' . $book_class . '.png';
          }
        }

        $temp_book = [
          'title' => $title,
          'cover_url' => $cover_url,
          'details_url' => $details_url,
        ];

        array_push($books_display, $temp_book);
      }
    }

    $result = [
      '#theme' => 'bibliocommons_books',
      '#parent_entity' => $parent_entity,
      '#books' => $books_display,
      '#title' => $list_name,
      '#show_more_link' => $show_more_link,
      '#cover_url' => '',
      '#details_url' => $list_details_url,
      '#type' => self::TYPE,
      '#attached' => [
        'library' => ['bibliocommons/bibliocommons'],
      ],
    ];

    return $result;
  }

}
