<?php

namespace Drupal\bigbluebutton\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\bigbluebutton\BBB;
use BigBlueButton\Parameters\GetRecordingsParameters;
use Drupal\Core\Url;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Access\AccessResult;


/**
 *
 */
class BBBDownloadRecordingForm extends FormBase {

  use LoggerChannelTrait;

  /**
   * @var array settings
   */
  protected $settings;  

  /**
   * @param array settings
   */
  public function __construct(array $settings = null) {
    $this->settings = $settings;
  }  

  /**
   * Get the recording ID of a recording and sends API Call to delete it.
   *
   * @return string
   *   The unique string identifying the form.
   */
  public function getFormId() {

    $form_id = 'bbb_download_recording';

    static $count = 0;
    $count++;

    return $form_id . '_' . $count;

  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $recording_id = NULL, $settings = NULL) {

    $this->settings = $settings; 

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['recording_id'] = [
      '#type' => 'hidden',
      '#value' => $recording_id,
    ];

    $form['actions']['download'] = [
      '#type' => 'submit',
      '#value' => t('Download'),
      '#attributes' => ['onclick' => 'if(!confirm("Are you sure you want to download that recording?")){return false;}'],
    ];

    return $form;

  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    $recording_id = $form_state->getValue('recording_id');

    if (isset($recording_id)) {

      $apiUrl = \Drupal::config('bigbluebutton.settings')->get('hostname');
      $secretKey = \Drupal::config('bigbluebutton.settings')->get('secret');
      $bbb = new BBB($secretKey, $apiUrl);

      $recordingParams = new GetRecordingsParameters();
      $recordingParams->setRecordID($recording_id);

      try {
        $response = $bbb->getRecordings($recordingParams);
        if (!empty($response->getRawXml()->recordings->recording)) {
          foreach ($response->getRawXml()->recordings->recording as $key => $recording) {
            foreach ($recording->playback as $key => $playback) {
              foreach ($recording->playback->format as $key => $format) {
                if ($format->type == "video_mp4" || $format->type == "video") {
                  $recording_url = $format->url->__toString();
                  $html = \Drupal::httpClient()->get($recording_url)->getBody()->getContents();
                  if (preg_match('/source\s+src="([^"]+\.(mp4|webm|ogg|m4v))"/i', $html, $matches)) {
                    $downloadUrl = $recording_url . $matches[1];
                  }

                  if (isset($downloadUrl)) {   
                    $url = Url::fromRoute('bigbluebutton.download_recording', [], [
                      'query' => ['url' => $downloadUrl],
                    ])->toString();
                    $form_state->setRedirectUrl(Url::fromUserInput($url));                     
                  }
                  else {
                    $this->getLogger('bigbluebutton')->warning('No downloadable format found for recording ID: @id', ['@id' => $recording_id]);
                    throw new \Exception('No downloadable format found.');
                  }
                }
              }
            }
          }
        }
        else {
          $error_message = $this->t("No recordings found for the given ID!");
          $this->messenger()->addError($error_message);
        }
      }
      catch (\RuntimeException $exception) {
        $this->getLogger('bigbluebutton')->warning($exception->getMessage());
        $error_message = $this->t("Couldn't get meeting info! please contact system administrator.");
        $this->messenger()->addError($error_message);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access(AccountInterface $account) {
    return AccessResult::allowedIfHasPermission($account, 'access bigbluebutton video download');
  }  

}