<?php

declare(strict_types=1);

namespace Drupal\bigbluebutton\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;


/**
 * Provides a Big Blue Button form.
 */
final class DefaultPresentationForm extends FormBase {


  /**
   * @var \Drupal\Core\KeyValueStore\KeyValueFactoryInterface
   */
  protected KeyValueFactoryInterface $kv;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('keyvalue')
    );
  }

  /**
   * @param \Drupal\Core\KeyValueStore\KeyValueFactoryInterface $key_value_factory
   */
  public function __construct(KeyValueFactoryInterface $key_value_factory) {
    $this->kv = $key_value_factory;
  }


  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'bigbluebutton_default_presentation';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    // Get stored file IDs from Key-Value storage.
    $file_id = $this->kv->get('bigbluebutton')->get('default_presentation', []);

    if (isset($file_id) && !empty($file_id)) {
      $default_presentation_default = $file_id;
    }
    else {
      $default_presentation_default = NULL; 
    }
  
    $validators = [
      'file_validate_extensions' => ['pdf'],
    ];    

    $form['default_presentation'] = [
      '#type' => 'managed_file',
      '#name' => 'default_presentation',
      '#title' => $this->t('Upload default presentation'),
      '#size' => '50',
      '#default_value' => $default_presentation_default,
      '#description' => $this->t('Add a default presentation as PDF to BigBlueButton.'),
      '#upload_validators' => $validators,
      '#upload_location' => 'public://bigbluebutton/',
    ];


    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save default presentation'),
        '#button_type' => 'primary',
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // @todo Validate the form here.
    // Example:
    // @code
    //   if (mb_strlen($form_state->getValue('message')) < 10) {
    //     $form_state->setErrorByName(
    //       'message',
    //       $this->t('Message should be at least 10 characters.'),
    //     );
    //   }
    // @endcode
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $file_id = $form_state->getValue('default_presentation');
    $this->kv->get('bigbluebutton')->set('default_presentation', $file_id);

    // Set the files as permanent.
    if (!empty($file_id)) {
      $file = File::load(reset($file_id));
      if ($file) {
        $file->setPermanent();
        $file->save();
        $this->messenger()
          ->addMessage($this->t('Default presentation for BigBlueButton has been saved.'));        
      }
    }
    //$form_state->setRedirect('bigbluebutton.settings');
  }

}
