<?php declare(strict_types = 1);

namespace Drupal\bigbluebutton\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Component\Utility\Environment;
use Drupal\file\Entity\File;

/**
 * Configure Bigbluebutton settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'bigbluebutton_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['bigbluebutton.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    $form['bbb_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('BBB Server settings'),
      '#open' => TRUE,
    ];

    $form['bbb_settings']['hostname'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Hostname'),
      '#default_value' => $this->config('bigbluebutton.settings')->get('hostname'),
      '#required' => TRUE
    ];
    $form['bbb_settings']['secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Secret'),
      '#default_value' => $this->config('bigbluebutton.settings')->get('secret'),
      '#required' => TRUE
    ];    
    
    $form['bbb_display'] = [
      '#type' => 'details',
      '#title' => $this->t('Display settings'),
    ];    

    $form['bbb_display']['user_display_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('User display name'),
      '#default_value' => $this->config('bigbluebutton.settings')->get('user_display_name'),
      '#description' => $this->t('The name that will be displayed in the BigBlueButton meeting. If left empty, the user\'s name will be used. Use tokens that can be resolved with the current users account'),
    ];

    $max_size = Environment::getUploadMaxSize();
    if (isset($max_size) && is_numeric($max_size)) {
      $max_size_mb = $max_size / 1024 / 1024;
    }
    else {
      $max_size_mb = 0;
    }    

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {   
    // If we have a valid url we have to check it realy exists!
    if (!\Drupal::service('bigbluebutton.helper')->isValidExternalURL($form_state->getValue('hostname'))) {
      $form_state->setErrorByName('hostname', $this->t('The entered URL is not valid!'));
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {  
    $this->config('bigbluebutton.settings')
      ->set('hostname', $form_state->getValue('hostname'))
      ->set('secret', $form_state->getValue('secret'))
      ->set('user_display_name', $form_state->getValue('user_display_name'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
