<?php declare(strict_types = 1);

namespace Drupal\bigbluebutton\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use BigBlueButton\Parameters\GetRecordingsParameters;
use Drupal\bigbluebutton\BBB;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Url;

/**
 * Plugin implementation of the 'Big Blue Button Recordings' formatter.
 *
 * @FieldFormatter(
 *   id = "bigbluebutton_recordings",
 *   label = @Translation("BBB Recordings"),
 *   field_types = {"bigbluebutton"},
 * )
 */
final class BigBlueButtonRecordingsFormatter extends FormatterBase {

  use MessengerTrait;
  use LoggerChannelTrait;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    $setting = [
      'display_options' => 'links',
      'supported_formats' => [], 
    ];
    return $setting + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $elements['display_options'] = [
      '#type' => 'select',
      '#title' => $this->t('Display options'),
      '#options' => ['thumbnail' => 'thumbnail', 'video' => 'video', 'links' => 'links', 'download_only' => 'download only'],
      '#default_value' => $this->getSetting('display_options'),
    ];

    $elements['supported_formats'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Supported recording formats'),
      '#description' => $this->t('Choose the formats to display. Leave unchecked to allow all formats.'),
      '#options' => [
        'presentation' => 'presentation',
        'screenshare' => 'screenshare',
        'video' => 'video',
        'video_mp4' => 'video_mp4',
      ],
      '#default_value' => $this->getSetting('supported_formats'),
    ];

    return $elements;
  }  

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $element = [];

    $item = $items[0];

    $display_options = $this->getSetting('display_options');

    $entity = $items->getEntity();

    $settings = [
      'display_options' => $this->getSetting('display_options'),
      'entity' => $items->getEntity(),
      'bbb' => []
    ];

    $allowed_video_types = [
      'video_mp4',
      'video',
    ];

    $apiUrl = \Drupal::config('bigbluebutton.settings')->get('hostname');
    $secretKey = \Drupal::config('bigbluebutton.settings')->get('secret');
    $bbb = new BBB($secretKey, $apiUrl);

    $recordingParams = new GetRecordingsParameters();
    $recordingParams->setMeetingID($entity->uuid());

    // Get current user
    $current_user = \Drupal::currentUser();


    try {
      $response = $bbb->getRecordings($recordingParams);
      if (!empty($response->getRawXml()->recordings->recording)) {
        $i = 1;
        $recordings = [];
        foreach ($response->getRawXml()->recordings->recording as $key => $recording) {
          foreach ($recording->playback as $key => $playback) {
            foreach ($recording->playback->format as $key => $format) {

              $supported_formats = array_filter($this->getSetting('supported_formats'));

              if (empty($supported_formats) || in_array((string) $format->type, $supported_formats, TRUE)) {
                
                $format->recordID = $recording->recordID;
                $recording_id = $recording->recordID->__toString();
                $recording_date = \Drupal::service('bigbluebutton.helper')->formatRecordingDate($recording_id);
                $delete_form = FALSE;
                $download_form = FALSE;
                if ($current_user->hasPermission('delete bigbluebutton recording')) {
                  $delete_form = \Drupal::formBuilder()->getForm('\Drupal\bigbluebutton\Form\BBBDeleteRecordingForm', $recording_id, $settings);
                }

                // Download video should available for videos only
                if ($format->type == "video_mp4" || $format->type == "video") {
                  if ($current_user->hasPermission('access bigbluebutton video download')) {
                    $download_form = \Drupal::formBuilder()->getForm('\Drupal\bigbluebutton\Form\BBBDownloadRecordingForm', $recording_id, $settings);
                  }                  
                }

                // Check if the user has permission to access the recording
                if ($current_user->hasPermission('access bigbluebutton recording')) {
                  // Add the recording to the recordings array                  
                  $recordings[] = [
                    'recording_id' => $recording_id,
                    'recording' => $format,
                    'recording_position' => $i,
                    'recording_link' => $recording->url->__toString(),
                    'recording_date' => $recording_date,
                    'delete_form' => $delete_form,
                    'download_form' => $download_form,
                    'recording_type' => $format->type,
                  ];                
                  $i = $i + 1;
                }
              }
            }
          }
        } 

        switch ($display_options) {
          case 'links':
            $element[0] = [
              '#theme' => 'bbb_recordings_links',
              '#url' => Url::fromRoute('bigbluebutton.view_recording', ['entity_uuid' => $entity->uuid()]),
              '#display_options' => $display_options,
              '#recordings' => $recordings,
            ];
            break;

          case 'thumnails':
            $element[0] = [
              '#theme' => 'bbb_recordings_thumbnails',
              '#url' => Url::fromRoute('bigbluebutton.view_recording', ['entity_uuid' => $entity->uuid()]),
              '#display_options' => $display_options,
              '#recordings' => $recordings,
            ];
            break;

          case 'video':
            $element[0] = [
              '#theme' => 'bbb_recordings_video',
              '#recordings' => $recordings,
            ];
            break;

          case 'download_only':
            // Filter recordings to only include video types
            foreach ($recordings as $index => $rec) {
              if (!in_array($rec['recording_type'], $allowed_video_types)) {
                unset($recordings[$index]);
              }
            }                        

            $element[0] = [
              '#theme' => 'bbb_recordings_download_only',
              '#url' => Url::fromRoute('bigbluebutton.view_recording', ['entity_uuid' => $entity->uuid()]),
              '#display_options' => $display_options,
              '#recordings' => $recordings,
            ];
            break;

          default:
            $element[0] = [
              '#theme' => 'bbb_recordings_links',
              '#url' => Url::fromRoute('bigbluebutton.view_recording', ['entity_uuid' => $entity->uuid()]),
              '#display_options' => $display_options,
              '#recordings' => $recordings,
            ];
            break;
        }      
      }
    }
    catch (\RuntimeException $exception) {
      $this->getLogger('bigbluebutton')->warning($exception->getMessage());
      $error_message = $this->t("Couldn't get recordings! please contact system administrator.");
      $this->messenger()->addError($error_message);
    }

    return $element;

  }

}
