<?php

namespace Drupal\bigbluebutton\Controller;

use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use GuzzleHttp\ClientInterface;

class BigBlueButtonDownloadRecordingController extends ControllerBase {

  protected $httpClient;
  protected $fileSystem;

  public function __construct(ClientInterface $http_client, FileSystemInterface $file_system) {
    $this->httpClient = $http_client;
    $this->fileSystem = $file_system;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('http_client'),
      $container->get('file_system')
    );
  }

  public function download() {
    // Normally you'd use the recording ID to get the real URL from BBB API.
    // For demo, we'll construct a mock one:
    $request = \Drupal::request();
    $video_url = $request->query->get('url');

    if (empty($video_url) || !filter_var($video_url, FILTER_VALIDATE_URL)) {
      throw new BadRequestHttpException('Missing or invalid "url" parameter.');
    }

    try {
      $filename = basename(parse_url($video_url, PHP_URL_PATH));
      $temp_uri = 'temporary://' . $filename;

      // Convert to real path for BinaryFileResponse
      $realPath = \Drupal::service('file_system')->realpath($temp_uri);    

      $response = $this->httpClient->get($video_url, ['sink' => $realPath]);
      if ($response->getStatusCode() !== 200) {
        throw new \Symfony\Component\HttpKernel\Exception\HttpException(500, 'Failed to download the recording file.');
      }

      $response = new BinaryFileResponse($realPath);
      $response->setContentDisposition(
        ResponseHeaderBag::DISPOSITION_ATTACHMENT,
        $filename
      );

      // Detect and set MIME type
      $finfo = finfo_open(FILEINFO_MIME_TYPE);
      $mime = finfo_file($finfo, $realPath);
      finfo_close($finfo);
      $response->headers->set('Content-Type', $mime);

      return $response;
      
    }
    catch (\Exception $e) {
      \Drupal::logger('bigbluebutton')->error('Download failed: @message', ['@message' => $e->getMessage()]);
      throw new BadRequestHttpException('Failed to download recording.');
    }
  }
}
