<?php

namespace Drupal\block_backup_restore\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\Environment;
use Drupal\Core\Url;

/**
 * Provides a form for restoring block configuration from a backup.
 *
 * This form allows users to upload and restore block configuration settings,
 * facilitating backup and recovery operations for block configurations.
 */
class RestoreConfigForm extends ConfigFormBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityManager;
  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Creates a RestoreConfigForm.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityManager
   *   The entity type manager service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    EntityTypeManagerInterface $entityManager,
    MessengerInterface $messenger,
  ) {
    $this->entityManager = $entityManager;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'block_backup_restore.setting',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'block_restore_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $theme = NULL) {
    // Form constructor.
    $form = parent::buildForm($form, $form_state);

    $form['block_restore'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Restore Block Layout Configuration by Using JSON File.'),
    ];

    $validators = [
      'FileExtension' => ['extensions' => 'json'],
      'FileSizeLimit' => ['fileLimit' => Environment::getUploadMaxSize()],
    ];

    // Hidden validation field.
    $form['block_restore']['theme_name'] = [
      '#type' => 'hidden',
      '#value' => $theme,
    ];
    // File Field for the JSON File.
    $form['block_restore']['json_file'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('JSON File'),
      '#upload_location' => 'public://block_restore/',
      '#default_value' => [],
      '#description' => [
        '#theme' => 'file_upload_help',
        '#description' => $this->t('The JSON file must include columns in the following order:'),
        '#upload_validators' => $validators,
      ],
      '#upload_validators' => $validators,
    ];

    $form['actions']['submit']['#value'] = $this->t('Restore layout');

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $fid = $form_state->getValue('json_file');

    if (empty($fid)) {
      $form_state->setErrorByName('json_file', $this->t('Please upload a JSON file.'));
      return;
    }

    // Load the file entity.
    $file = $this->entityManager->getStorage('file')->load($fid[0]);

    if (!$file) {
      $form_state->setErrorByName('json_file', $this->t('The uploaded file could not be loaded.'));
    }
    else {
      // Mark file as permanent.
      $file->setPermanent();
      $file->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if ($form_state->getValue('json_file')) {
      $fileId = $form_state->getValue('json_file')[0];
      $fileObject = $this->entityManager->getStorage('file')->load($fileId);
      $fileData = file_get_contents($fileObject->getFileUri());
      if (!empty($fileData)) {
        $json = json_decode($fileData, TRUE);
        foreach ($json as $key => $value) {
          if (!empty($key)) {
            $blockData = $this->entityManager->getStorage('block')->load($key);
            if (!empty($blockData)) {
              $blockData->setRegion($value['region'])
                ->setWeight($value['weight'])
                ->enable()
                ->save();
            }
          }
        }
      }
    }
    $this->messenger->addMessage($this->t('Block Restore in the @themeName theme successfully.', ['@themeName' => ucwords($form_state->getValue('theme_name'))]));
    $url = Url::fromRoute('block.admin_display_theme', ['theme' => $form_state->getValue('theme_name')]);
    $form_state->setRedirectUrl($url);
  }

}
