<?php

namespace Drupal\block_content_type_visibility\Plugin\Condition;

use Drupal\Core\Condition\ConditionPluginBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a 'Content Type' condition with show/hide options.
 *
 * @Condition(
 *   id = "content_type_visibility",
 *   label = @Translation("Content Type"),
 *   context_definitions = {
 *     "node" = @ContextDefinition("entity:node", label = @Translation("Node"), required = FALSE)
 *   }
 * )
 */
class ContentType extends ConditionPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The entity storage for node types.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected EntityStorageInterface $nodeTypeStorage;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * Constructs a ContentType condition plugin.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $node_type_storage
   *   The entity storage for node types.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityStorageInterface $node_type_storage,
    RouteMatchInterface $route_match,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->nodeTypeStorage = $node_type_storage;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')->getStorage('node_type'),
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'content_types' => [],
      'visibility_mode' => 'show',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Load all content types.
    $node_types = $this->nodeTypeStorage->loadMultiple();
    $options = [];

    foreach ($node_types as $type) {
      $options[$type->id()] = $type->label();
    }

    $form['visibility_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Visibility mode'),
      '#options' => [
        'show' => $this->t('Show for the selected content types'),
        'hide' => $this->t('Hide for the selected content types'),
      ],
      '#default_value' => $this->configuration['visibility_mode'],
      '#description' => $this->t('Choose whether to show or hide the block for the selected content types.'),
    ];

    $form['content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Content types'),
      '#options' => $options,
      '#default_value' => $this->configuration['content_types'],
      '#description' => $this->t('Select the content types for which this condition applies.'),
    ];

    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    // Filter out empty values from checkboxes.
    $content_types = array_filter($form_state->getValue('content_types'));
    $this->configuration['content_types'] = $content_types;
    $this->configuration['visibility_mode'] = $form_state->getValue('visibility_mode');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function summary() {
    $content_types = $this->configuration['content_types'];
    $mode = $this->configuration['visibility_mode'];

    if (empty($content_types)) {
      return $this->t('No content types selected');
    }

    // Load node type labels.
    $node_types = $this->nodeTypeStorage->loadMultiple($content_types);
    $type_labels = [];
    foreach ($node_types as $type) {
      $type_labels[] = $type->label();
    }

    $types_string = implode(', ', $type_labels);

    if ($mode === 'show') {
      return $this->t('Show for content types: @types', ['@types' => $types_string]);
    }
    else {
      return $this->t('Hide for content types: @types', ['@types' => $types_string]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function evaluate() {
    // If no content types are selected, block should not be shown.
    if (empty($this->configuration['content_types'])) {
      return FALSE;
    }

    // Try to get node from context.
    $node = $this->getContextValue('node');

    // If no node in context, try to get from route.
    if (!$node) {
      $node = $this->routeMatch->getParameter('node');
    }

    // If still no node, return FALSE.
    if (!$node) {
      return FALSE;
    }

    // Check if the current node type is in the configured types.
    $current_type = $node->getType();
    $is_selected_type = in_array($current_type, $this->configuration['content_types'], TRUE);

    // Return based on visibility mode.
    if ($this->configuration['visibility_mode'] === 'show') {
      // Show mode: return TRUE if type is selected.
      return $is_selected_type;
    }
    else {
      // Hide mode: return TRUE if type is NOT selected.
      return !$is_selected_type;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    $contexts = parent::getCacheContexts();
    // Add route cache context since we check the current node.
    $contexts[] = 'route';
    return $contexts;
  }

}
