<?php

namespace Drupal\Tests\block_content_type_visibility\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;
use Drupal\node\Entity\Node;

/**
 * Tests the content type visibility condition plugin.
 *
 * @group block_content_type_visibility
 */
class ContentTypeConditionTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'block',
    'block_content_type_visibility',
  ];

  /**
   * The condition plugin manager.
   *
   * @var \Drupal\Core\Condition\ConditionManager
   */
  protected $pluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['node', 'system']);

    // Create node types.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    NodeType::create([
      'type' => 'page',
      'name' => 'Page',
    ])->save();

    $this->pluginManager = $this->container->get('plugin.manager.condition');
  }

  /**
   * Tests the content type condition with show mode.
   */
  public function testShowMode() {
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    $condition = $this->pluginManager->createInstance('content_type_visibility');
    $condition->setConfiguration([
      'content_types' => ['article'],
      'visibility_mode' => 'show',
    ]);
    $condition->setContextValue('node', $node);

    $this->assertTrue($condition->execute(), 'Block should be shown for article content type.');

    // Test with different content type.
    $page = Node::create([
      'type' => 'page',
      'title' => 'Test Page',
    ]);
    $page->save();

    $condition->setContextValue('node', $page);
    $this->assertFalse($condition->execute(), 'Block should not be shown for page content type.');
  }

  /**
   * Tests the content type condition with hide mode.
   */
  public function testHideMode() {
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    $condition = $this->pluginManager->createInstance('content_type_visibility');
    $condition->setConfiguration([
      'content_types' => ['article'],
      'visibility_mode' => 'hide',
    ]);
    $condition->setContextValue('node', $node);

    $this->assertFalse($condition->execute(), 'Block should be hidden for article content type.');

    // Test with different content type.
    $page = Node::create([
      'type' => 'page',
      'title' => 'Test Page',
    ]);
    $page->save();

    $condition->setContextValue('node', $page);
    $this->assertTrue($condition->execute(), 'Block should be shown for page content type.');
  }

  /**
   * Tests the condition with multiple content types.
   */
  public function testMultipleContentTypes() {
    $article = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $article->save();

    $page = Node::create([
      'type' => 'page',
      'title' => 'Test Page',
    ]);
    $page->save();

    $condition = $this->pluginManager->createInstance('content_type_visibility');
    $condition->setConfiguration([
      'content_types' => ['article', 'page'],
      'visibility_mode' => 'show',
    ]);

    $condition->setContextValue('node', $article);
    $this->assertTrue($condition->execute(), 'Block should be shown for article content type.');

    $condition->setContextValue('node', $page);
    $this->assertTrue($condition->execute(), 'Block should be shown for page content type.');
  }

  /**
   * Tests the condition with no content types selected.
   */
  public function testNoContentTypesSelected() {
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
    ]);
    $node->save();

    $condition = $this->pluginManager->createInstance('content_type_visibility');
    $condition->setConfiguration([
      'content_types' => [],
      'visibility_mode' => 'show',
    ]);
    $condition->setContextValue('node', $node);

    $this->assertFalse($condition->execute(), 'Block should not be shown when no content types are selected.');
  }

}
