<?php

declare(strict_types=1);

namespace Drupal\block_content_type_visibility\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;

/**
 * Hook implementations for block_content_type_visibility module.
 */
final class BlockContentTypeVisibilityHooks {

  /**
   * Implements hook_form_FORM_ID_alter() for block_form.
   *
   * Adds user-friendly visibility mode radio buttons to replace the
   * confusing "negate" checkbox for content type visibility conditions.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param string $form_id
   *   The form ID.
   */
  #[Hook('form_block_form_alter')]
  public function formBlockFormAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Check if the entity_bundle:node visibility condition exists.
    if (isset($form['visibility']['entity_bundle:node'])) {
      $block = $form_state->getFormObject()->getEntity();
      $config = $block->getVisibilityConditions()->getConfiguration();

      // Get current configuration for entity_bundle:node.
      $current_config = $config['entity_bundle:node'] ?? [];
      $negate = $current_config['negate'] ?? FALSE;

      // Add visibility mode radio buttons at the top.
      $form['visibility']['entity_bundle:node']['visibility_mode'] = [
        '#type' => 'radios',
        '#title' => t('Visibility mode'),
        '#options' => [
          'show' => t('Show for the selected content types'),
          'hide' => t('Hide for the selected content types'),
        ],
        '#default_value' => $negate ? 'hide' : 'show',
        '#description' => t('Choose whether to show or hide the block for the selected content types.'),
        '#weight' => -10,
      ];

      // Hide the original negate checkbox since we're replacing it.
      if (isset($form['visibility']['entity_bundle:node']['negate'])) {
        $form['visibility']['entity_bundle:node']['negate']['#access'] = FALSE;
      }

      // Add custom submit handler to convert visibility_mode to negate.
      $form['#submit'][] = [$this, 'formBlockFormSubmit'];
    }
  }

  /**
   * Custom submit handler for block form.
   *
   * Converts the visibility_mode radio selection to the negate boolean
   * that Drupal's visibility condition system expects.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function formBlockFormSubmit(array $form, FormStateInterface $form_state): void {
    $visibility = $form_state->getValue('visibility');

    if (isset($visibility['entity_bundle:node']['visibility_mode'])) {
      $visibility_mode = $visibility['entity_bundle:node']['visibility_mode'];

      // Convert visibility_mode to negate value.
      // 'show' = negate FALSE, 'hide' = negate TRUE.
      $negate_value = ($visibility_mode === 'hide') ? 1 : 0;
      $visibility['entity_bundle:node']['negate'] = $negate_value;

      // Remove visibility_mode from the saved configuration.
      unset($visibility['entity_bundle:node']['visibility_mode']);

      $form_state->setValue('visibility', $visibility);
    }
  }

}
