# AI Instructions for Block Editor Module

This is the Drupal Block Editor module, which integrates the Gutenberg block editor with Drupal CMS using a sophisticated Webpack-based build system.

## Key Guidelines for AI Assistance:

### Coding Standards
- **PHP**: Follow Drupal coding standards and PSR-4 autoloading
- **JavaScript**: Use ES6+ with React patterns
- **CSS/SCSS**: Use BEM naming convention, compile with webpack
- **Module Structure**: Maintain standard Drupal module structure

### Build System Architecture

#### **CRITICAL: Auto-Generated Files**
The following files are **auto-generated** by webpack and should **NEVER be edited manually**:
- `block_editor.libraries.yml` - Generated from build artifacts
- `build/` directory - All contents are build outputs
- `*.asset.php` files - Generated dependency manifests

#### Webpack Configuration Structure

1. **tools/webpack/packages.js**
   - Builds custom packages from `packages/` directory
   - Generates asset.php files with dependencies
   - Uses DependencyExtractionWebpackPlugin

2. **tools/webpack/vendors.js**
   - Copies React, ReactDOM, and vendor libraries to `build/vendors/`
   - No transpilation, just copying

3. **tools/webpack/wordpress-packages.js** ⭐ **Most Important**
   - Dynamically builds WordPress packages from package.json dependencies
   - **Dynamic Dependency Extraction**: Reads each package's package.json
   - Creates both production (minified) and development (source maps) builds
   - Generates custom asset.php files with proper dependencies
   - Maps: react → 'react', react-dom → 'react-dom', @wordpress/* → 'wp-*'

4. **tools/webpack/generate-libraries.js** ⭐ **Auto-Generator**
   - Custom webpack plugin that generates block_editor.libraries.yml
   - Scans build directory for asset.php files
   - Converts dependencies: wp-* → block_editor/wordpress_*, react → block_editor/react
   - Filters out self-references to prevent circular dependencies
   - Detects CSS files (both build/block_editor.css and fallback)

5. **webpack.config.js**
   - Main entry point configuration
   - Babel loader for JSX with automatic runtime
   - SCSS/SASS processing with PostCSS and autoprefixer
   - CSS extraction using MiniCssExtractPlugin
   - Externals: Drupal, @blockeditor/editor

### Adding WordPress Packages

When a user wants to add a new WordPress package:

1. Add to package.json dependencies (NOT devDependencies)
2. Run `npm install`
3. Run `npm run build`
4. The build system automatically:
   - Detects the package in package.json
   - Reads its dependencies from node_modules/@wordpress/[package]/package.json
   - Builds it to build/gutenberg/[package]/
   - Creates asset.php with extracted dependencies
   - Adds it to block_editor.libraries.yml

**Example:**
```bash
npm install @wordpress/components
npm run build
# Now available as block_editor/wordpress_components
```

### WordPress Package Dependency Extraction

The system uses `extractPackageDependencies()` function that:
- Reads node_modules/@wordpress/[package]/package.json
- Extracts both dependencies and peerDependencies
- Converts to proper format:
  - `react` → `'react'`
  - `react-dom` → `'react-dom'`
  - `@wordpress/element` → `'wp-element'`
- Returns array for asset.php generation

### Library Generation Logic

The GenerateDrupalLibrariesPlugin:
1. Scans build/vendors/ for vendor libraries
2. Scans build/gutenberg/ for WordPress packages
3. Scans build/editor/ for custom packages
4. Reads each .asset.php file to extract dependencies
5. Maps dependencies to Drupal library format
6. Generates YAML with proper structure and comments

### CSS/SCSS Processing

- SCSS files are processed with: sass-loader → postcss-loader (autoprefixer) → css-loader → MiniCssExtractPlugin
- CSS is extracted to `build/block_editor.css`
- Automatically added to libraries.yml if file exists
- Import in JS: `import './style.scss'`

### Important Patterns

#### Adding External Libraries
In webpack.config.js externals:
```javascript
externals: {
  'package-name': 'GlobalVariable',
}
```

#### Creating Custom Packages
1. Create in `packages/[name]/`
2. Add package.json with proper name
3. Build system will auto-detect and build
4. Will appear in libraries.yml

#### Debugging Build Issues
- Check console for "✓ Built WordPress packages to build/gutenberg/"
- Verify asset.php files have correct dependencies
- Check libraries.yml generation (should show "✓ Generated block_editor.libraries.yml")
- For missing deps, verify package.json (dependencies, not devDependencies)

### Module Purpose
The Block Editor module allows content creators to use Gutenberg blocks within Drupal, providing a modern editing experience with React-based components.

### Important Files Reference

**PHP:**
- `src/Form/EntityEditForm.php` - Main form with isolated block editor
- `block_editor.module` - Module hooks and functions

**JavaScript Build:**
- `js/index.js` - Main entry point
- `js/style.scss` - Main styles
- `packages/editor/` - Custom @blockeditor/editor package

**Build Configuration:**
- `webpack.config.js` - Main webpack config
- `tools/webpack/*.js` - Modular webpack configs
- `package.json` - Dependencies (check dependencies, NOT devDependencies for WordPress packages)

**Auto-Generated (DO NOT EDIT):**
- `block_editor.libraries.yml` - Generated by webpack
- `build/**` - All build outputs
- `**/*.asset.php` - Dependency manifests

### Common Tasks

**Add WordPress Package:**
```bash
npm install @wordpress/[package-name]
npm run build
```

**Add Custom Package:**
1. Create packages/[name]/
2. Add package.json
3. Run npm run build

**Update Styles:**
1. Edit js/style.scss
2. Run npm run build
3. CSS extracted to build/block_editor.css

**Debug Dependencies:**
1. Check build/gutenberg/[package]/index.min.asset.php
2. Verify node_modules/@wordpress/[package]/package.json
3. Check block_editor.libraries.yml generation

### Testing Guidelines
- Include PHPUnit tests for PHP code
- Test JavaScript builds with `npm run build`
- Verify library generation after changes
- Test in Drupal to ensure proper loading order

When making changes:
1. Never manually edit auto-generated files
2. Update webpack configs for build changes
3. Run full build to verify
4. Check that libraries.yml has correct dependencies
5. Test in Drupal environment
6. Update this documentation if patterns change
