# Build System Documentation

## Overview

The Block Editor module uses a sophisticated Webpack-based build system that automatically manages JavaScript dependencies, CSS processing, and Drupal library generation.

## Quick Start

```bash
# Install dependencies
npm install

# Build everything
npm run build

# Build only custom packages and vendors
npm run build:packages
```

## Build Architecture

### Build Pipeline

```
npm run build
    ↓
1. npm run build:packages (custom packages + vendors)
    ↓
2. wp-scripts build (main webpack config)
    ↓
3. Multiple webpack configurations execute:
    ├── packages.js → builds @blockeditor/* packages
    ├── vendors.js → copies React/ReactDOM vendors
    ├── wordpress-packages.js → builds @wordpress/* packages
    └── webpack.config.js → builds main entry + generates libraries.yml
```

### Output Structure

```
build/
├── vendors/                    # Vendor libraries (React, ReactDOM)
│   ├── react.min.js
│   ├── react.js
│   ├── react-dom.min.js
│   ├── react-dom.js
│   ├── react-jsx-runtime.min.js
│   └── react-jsx-runtime.js
│
├── gutenberg/                  # WordPress packages (auto-built from package.json)
│   ├── element/
│   │   ├── index.min.js       # Production build
│   │   ├── index.js           # Development build (with source maps)
│   │   ├── index.min.asset.php # Dependencies manifest
│   │   ├── index.asset.php
│   │   └── index.js.map
│   └── escape-html/
│       ├── index.min.js
│       ├── index.js
│       ├── index.min.asset.php
│       ├── index.asset.php
│       └── index.js.map
│
├── editor/                     # Custom @blockeditor packages
│   ├── index.min.js
│   └── index.min.asset.php
│
├── block_editor.js            # Main entry bundle
└── block_editor.css           # Extracted styles
```

## WordPress Package System

### How It Works

1. **Package Detection**
   - Scans `package.json` dependencies for `@wordpress/*` packages
   - Only uses `dependencies`, NOT `devDependencies`

2. **Dynamic Dependency Extraction**
   ```javascript
   // From tools/webpack/wordpress-packages.js
   function extractPackageDependencies(packageName) {
     const pkgJson = require(`@wordpress/${packageName}/package.json`);
     // Extracts dependencies and peerDependencies
     // Converts @wordpress/* → wp-*, react → react, etc.
   }
   ```

3. **Asset File Generation**
   - Creates `.asset.php` files with dependency arrays
   - Example: `['dependencies' => array('react', 'react-dom', 'wp-escape-html')]`

4. **Webpack Externals**
   - React and ReactDOM are externalized (not bundled)
   - Other WordPress packages can be bundled or externalized

### Adding WordPress Packages

```bash
# Add to dependencies
npm install @wordpress/components

# Build
npm run build
```

**Result:**
- Built to `build/gutenberg/components/`
- Dependencies extracted from package.json
- Added to `block_editor.libraries.yml` as `block_editor/wordpress_components`
- All dependencies automatically mapped

### Package Dependency Mapping

The build system automatically converts:

| Source Dependency | Asset File Format | Drupal Library Format |
|------------------|-------------------|----------------------|
| `react` | `'react'` | `block_editor/react` |
| `react-dom` | `'react-dom'` | `block_editor/react_dom` |
| `@wordpress/element` | `'wp-element'` | `block_editor/wordpress_element` |
| `@wordpress/i18n` | `'wp-i18n'` | `block_editor/wordpress_i18n` |

## Library Auto-Generation

### How Libraries.yml is Generated

The `GenerateDrupalLibrariesPlugin` (in `tools/webpack/generate-libraries.js`):

1. **Scans Build Directory**
   ```
   build/
   ├── vendors/     → Vendor libraries section
   ├── gutenberg/   → WordPress packages section  
   └── editor/      → Custom packages section
   ```

2. **Reads Asset Files**
   - Parses PHP asset files: `<?php return array('dependencies' => [...], 'version' => '...');`
   - Extracts dependency arrays

3. **Converts Dependencies**
   - `wp-*` → `block_editor/wordpress_*`
   - `react` → `block_editor/react`
   - `react-dom` → `block_editor/react_dom`

4. **Filters Self-References**
   - Removes circular dependencies automatically

5. **Generates YAML**
   ```yaml
   wordpress_element:
     js:
       build/gutenberg/element/index.min.js: {}
     dependencies:
       - block_editor/wordpress_escape_html
       - block_editor/react
       - block_editor/react_dom
   ```

### Library Sections

**Auto-generated sections:**

1. **Vendor Libraries**
   - `react`, `react_dom`, `react_jsx_runtime`
   - Hardcoded from build/vendors/

2. **WordPress Packages**
   - All `@wordpress/*` packages from dependencies
   - Dynamic based on asset.php files

3. **Custom Packages**
   - Packages from `packages/` directory
   - E.g., `@blockeditor/editor`

4. **Main Libraries**
   - `block_editor_form` with main JS and CSS
   - Dependencies on other libraries

## CSS Processing

### SCSS/SASS Pipeline

```
import './style.scss'
    ↓
sass-loader (SCSS → CSS)
    ↓
postcss-loader (autoprefixer)
    ↓
css-loader (resolve imports, urls)
    ↓
MiniCssExtractPlugin (extract to file)
    ↓
build/block_editor.css
```

### CSS Configuration

In `webpack.config.js`:

```javascript
{
  test: /\.(scss|sass)$/,
  use: [
    MiniCssExtractPlugin.loader,  // Extract CSS
    'css-loader',                 // Handle imports
    {
      loader: 'postcss-loader',
      options: {
        postcssOptions: {
          plugins: [require('autoprefixer')],
        },
      },
    },
    'sass-loader',                // Compile SCSS
  ],
}
```

### CSS in Libraries

The library generator automatically detects CSS:

```javascript
// Checks for build/block_editor.css
if (fs.existsSync(generatedCssFile)) {
  blockEditorLib.css = {
    theme: {
      'build/block_editor.css': {},
    },
  };
}
```

## Custom Package Development

### Creating a New Package

1. **Create Package Structure**
   ```
   packages/
   └── my-package/
       ├── package.json
       ├── src/
       │   └── index.js
       └── build-module/
           └── index.js
   ```

2. **Package.json**
   ```json
   {
     "name": "@blockeditor/my-package",
     "version": "1.0.0",
     "main": "build-module/index.js",
     "dependencies": {
       "react": "^18.0.0"
     }
   }
   ```

3. **Build**
   ```bash
   npm run build
   ```

4. **Use in Drupal**
   ```yaml
   # Auto-generated in block_editor.libraries.yml
   my_package:
     js:
       build/my-package/index.min.js: {}
     dependencies:
       - block_editor/react
   ```

## Webpack Configurations

### 1. packages.js - Custom Packages

- Builds packages from `packages/` directory
- Uses Babel for transpilation
- Generates asset.php files
- Uses DependencyExtractionWebpackPlugin

### 2. vendors.js - Vendor Libraries

- Copies React, ReactDOM to build/vendors/
- No transpilation
- Both development and production versions

### 3. wordpress-packages.js - WordPress Packages ⭐

**Most sophisticated configuration:**

```javascript
// Dynamic package discovery
function getWordPressPackages() {
  const deps = packageJson.dependencies || {};
  return Object.keys(deps)
    .filter(dep => dep.startsWith('@wordpress/'))
    .map(dep => dep.replace('@wordpress/', ''));
}

// Dynamic dependency extraction
function extractPackageDependencies(packageName) {
  const pkgJson = require(`@wordpress/${packageName}/package.json`);
  const deps = [];
  
  Object.keys({ ...pkgJson.dependencies, ...pkgJson.peerDependencies })
    .forEach(dep => {
      if (dep === 'react') deps.push('react');
      else if (dep === 'react-dom') deps.push('react-dom');
      else if (dep.startsWith('@wordpress/')) {
        deps.push('wp-' + dep.replace('@wordpress/', ''));
      }
    });
  
  return deps;
}
```

Returns **two configs**:
- Production: minified with content hash
- Development: source maps for debugging

### 4. webpack.config.js - Main Entry

- Main entry point: `js/index.js`
- Babel loader with React automatic runtime
- SCSS/CSS processing
- Externals configuration
- Library generation plugin

## Debugging

### Build Issues

**Check build output:**
```bash
npm run build 2>&1 | grep -E "(Built|Generated|compiled)"
```

**Verify WordPress packages:**
```bash
npm run build 2>&1 | grep -A 5 "Built WordPress"
```

**Check generated files:**
```bash
# Asset files
cat build/gutenberg/element/index.min.asset.php

# Libraries
grep -A 5 "wordpress_element:" block_editor.libraries.yml
```

### Common Issues

1. **Package not building**
   - Check: Is it in `dependencies` (not `devDependencies`)?
   - Run: `npm install` then `npm run build`

2. **Wrong dependencies in asset.php**
   - Check: `node_modules/@wordpress/[package]/package.json`
   - Verify: Dependencies and peerDependencies

3. **Missing in libraries.yml**
   - Check: Does asset.php file exist?
   - Verify: Build completed successfully
   - Check: GenerateDrupalLibrariesPlugin output

4. **CSS not extracted**
   - Check: Is SCSS imported in JavaScript?
   - Verify: MiniCssExtractPlugin is configured
   - Check: build/block_editor.css exists

## Advanced Topics

### Adding External Dependencies

In `webpack.config.js`:

```javascript
externals: {
  'my-global': 'MyGlobal',  // window.MyGlobal
  '@custom/lib': 'CustomLib',
}
```

### Custom Asset Generation

Modify `tools/webpack/wordpress-packages.js`:

```javascript
class WordPressAssetPlugin {
  apply(compiler) {
    compiler.hooks.emit.tapAsync('WordPressAssetPlugin', (compilation, callback) => {
      // Custom logic here
    });
  }
}
```

### Extending Library Generation

Modify `tools/webpack/generate-libraries.js`:

```javascript
generateLibraries() {
  const libraries = {};
  
  // Add custom library sections
  libraries['# Custom Section'] = null;
  libraries.my_custom = { /* ... */ };
  
  return libraries;
}
```

## Performance

### Production Build

- Minification enabled
- Source maps disabled (only in dev builds)
- Content hash in version for cache busting

### Development Build

- Source maps for debugging
- Faster build times
- Non-minified for readability

### Build Time Optimization

- Webpack cache enabled
- Parallel processing for multiple configs
- Incremental builds when possible

## Best Practices

1. **Always use dependencies, not devDependencies** for WordPress packages
2. **Never edit auto-generated files** (libraries.yml, build/, *.asset.php)
3. **Run full build** after package.json changes
4. **Check build output** for errors and warnings
5. **Verify libraries.yml** after significant changes
6. **Test in Drupal** to ensure correct loading order
7. **Keep webpack configs modular** for maintainability
8. **Document custom changes** in this file

## Troubleshooting Commands

```bash
# Full rebuild
rm -rf build/ node_modules/
npm install
npm run build

# Check package detection
node -e "const pkg = require('./package.json'); console.log(Object.keys(pkg.dependencies).filter(d => d.startsWith('@wordpress/')))"

# Verify asset file
cat build/gutenberg/element/index.min.asset.php

# Check library generation
grep -A 20 "# WordPress packages" block_editor.libraries.yml

# Build with verbose output
npm run build 2>&1 | tee build.log
```
