# Build System Changelog

## Major Improvements - October 2025

### 🚀 Dynamic WordPress Package Integration

Implemented a sophisticated build system that automatically integrates WordPress Gutenberg packages with Drupal.

#### Key Features

1. **Automatic Package Detection**
   - Scans `package.json` dependencies for `@wordpress/*` packages
   - Only processes actual dependencies (not devDependencies)
   - No manual configuration needed

2. **Dynamic Dependency Extraction**
   - Reads each WordPress package's `package.json` at build time
   - Extracts both `dependencies` and `peerDependencies`
   - Automatically converts to proper format:
     - `react` → `'react'`
     - `react-dom` → `'react-dom'`
     - `@wordpress/element` → `'wp-element'`

3. **Automated Build Pipeline**
   - Production builds (minified) to `[package]/index.min.js`
   - Development builds (source maps) to `[package]/index.js`
   - Asset files with dependency manifests
   - All in `build/gutenberg/` directory

4. **Smart Library Generation**
   - Auto-generates `block_editor.libraries.yml` from build artifacts
   - Scans build directory for asset files
   - Converts dependencies to Drupal library format
   - Filters circular dependencies automatically
   - No manual YAML editing required

#### Usage

```bash
# Add any WordPress package
npm install @wordpress/components @wordpress/block-editor

# Build
npm run build

# Result: Packages are automatically:
# - Built to build/gutenberg/[package]/
# - Added to block_editor.libraries.yml
# - Available as block_editor/wordpress_[package]
```

### 🎨 CSS/SCSS Processing

Implemented complete SCSS processing pipeline:

1. **SCSS/SASS Compilation**
   - sass-loader for SCSS → CSS conversion
   - Support for .scss and .sass files

2. **PostCSS Processing**
   - autoprefixer for vendor prefixes
   - Configurable PostCSS plugins

3. **CSS Extraction**
   - MiniCssExtractPlugin extracts CSS to files
   - Output to `build/block_editor.css`
   - Automatically added to libraries.yml

4. **Import in JavaScript**
   ```javascript
   import './style.scss';
   ```

### 🔧 Webpack Architecture

Modular webpack configuration system:

1. **packages.js** - Custom package builder
   - Builds @blockeditor/* packages
   - DependencyExtractionWebpackPlugin integration

2. **vendors.js** - Vendor library copier
   - React, ReactDOM, react-jsx-runtime
   - Development and production versions

3. **wordpress-packages.js** - WordPress package builder ⭐
   - Dynamic package detection
   - Dependency extraction from package.json
   - Custom asset file generation
   - Dual build (prod + dev)

4. **generate-libraries.js** - Library auto-generator ⭐
   - Webpack plugin for libraries.yml generation
   - Scans build directory
   - Parses asset.php files
   - Converts to Drupal library format
   - Handles CSS detection

5. **webpack.config.js** - Main configuration
   - Entry point setup
   - Babel loader for JSX
   - SCSS/CSS rules
   - Externals configuration
   - Plugin orchestration

### 📋 Auto-Generated Files

The following files are now **completely auto-generated** and should never be edited manually:

- `block_editor.libraries.yml` - Generated by webpack build
- `build/**` - All build outputs
- `**/*.asset.php` - Dependency manifests

### 🧪 Examples

#### Example 1: Adding @wordpress/element

```bash
npm install @wordpress/element
npm run build
```

**Result:**
```php
// build/gutenberg/element/index.min.asset.php
<?php return array(
  'dependencies' => array('react', 'react-dom'),
  'version' => 'abe3de9ec4f89517b8b4'
);
```

```yaml
# block_editor.libraries.yml
wordpress_element:
  js:
    build/gutenberg/element/index.min.js: {}
  dependencies:
    - block_editor/react
    - block_editor/react_dom
```

#### Example 2: Adding @wordpress/components

```bash
npm install @wordpress/components
npm run build
```

**Auto-detected dependencies from @wordpress/components/package.json:**
- react
- react-dom
- @wordpress/element
- @wordpress/i18n
- @wordpress/compose

**Generated library:**
```yaml
wordpress_components:
  js:
    build/gutenberg/components/index.min.js: {}
  dependencies:
    - block_editor/react
    - block_editor/react_dom
    - block_editor/wordpress_element
    - block_editor/wordpress_i18n
    - block_editor/wordpress_compose
```

### 🔍 Technical Details

#### Dependency Extraction Algorithm

```javascript
function extractPackageDependencies(packageName) {
  const pkgJson = require(`@wordpress/${packageName}/package.json`);
  const allDeps = {
    ...pkgJson.dependencies,
    ...pkgJson.peerDependencies
  };
  
  const deps = [];
  Object.keys(allDeps).forEach(dep => {
    if (dep === 'react') deps.push('react');
    else if (dep === 'react-dom') deps.push('react-dom');
    else if (dep.startsWith('@wordpress/')) {
      deps.push('wp-' + dep.replace('@wordpress/', ''));
    }
  });
  
  return deps;
}
```

#### Asset File Generation

```javascript
class WordPressAssetPlugin {
  apply(compiler) {
    compiler.hooks.emit.tapAsync('WordPressAssetPlugin', (compilation, callback) => {
      packages.forEach(pkg => {
        const deps = extractPackageDependencies(pkg);
        const hash = chunk.contentHash.javascript;
        const depsArray = deps.map(dep => `'${dep}'`).join(', ');
        const content = `<?php return array('dependencies' => array(${depsArray}), 'version' => '${hash}');`;
        
        compilation.assets[`${pkg}/index.min.asset.php`] = {
          source: () => content,
          size: () => content.length
        };
      });
      callback();
    });
  }
}
```

### 📚 Documentation Updates

Created comprehensive documentation:

1. **README.md** - User-facing documentation
   - Installation instructions
   - Build system overview
   - WordPress package integration guide
   - Development workflow

2. **.ai-instructions.md** - AI assistant guidelines
   - Architecture explanation
   - Critical patterns and rules
   - Auto-generated file warnings
   - Common tasks and debugging

3. **BUILD.md** - Technical build documentation
   - Detailed architecture
   - Webpack configuration breakdown
   - Dependency mapping tables
   - Troubleshooting guide
   - Advanced topics

### 🎯 Benefits

1. **Zero Configuration** - Add WordPress packages with npm install, build automatically
2. **Always Accurate** - Dependencies extracted from source package.json files
3. **No Manual Maintenance** - Libraries.yml auto-generated on every build
4. **Proper Loading Order** - Drupal dependency chains created automatically
5. **Development Friendly** - Source maps, hot reload ready, clear error messages
6. **Production Ready** - Minification, content hashing, optimized output

### 🔄 Migration Notes

- Old hardcoded dependency map removed
- All libraries.yml editing is now automated
- CSS moved from inline injection to file extraction
- Build output structure standardized

### 🚦 Testing

All features tested with:
- @wordpress/element - Base React wrapper
- @wordpress/escape-html - Utility package (no deps)
- Multiple package dependency chains
- CSS extraction and library integration
- Asset file generation and parsing
