# Block Editor Module

## INTRODUCTION

The Block Editor module integrates a modern block-based editing experience into Drupal, providing an isolated form editing environment with React-based components and WordPress Gutenberg package support.

The primary use case for this module is:

- Provide a block-based editing interface for Drupal content
- Integrate WordPress Gutenberg packages with Drupal
- Enable isolated form editing with modern JavaScript frameworks
- Support custom block development and extensibility

## REQUIREMENTS

- Drupal 10.x
- Node.js 18+ and npm (for development)
- PHP 8.1+

## INSTALLATION

1. Install the module as you would normally install a contributed Drupal module.
   See: https://www.drupal.org/node/895232 for further information.

2. For development, install JavaScript dependencies:
   ```bash
   cd web/modules/custom/block_editor
   npm install
   ```

## DEVELOPMENT

### Build System

The module uses a sophisticated Webpack-based build system with the following features:

#### Build Commands

```bash
# Build all assets (packages, vendors, WordPress packages, and main bundle)
npm run build

# Build only custom packages and vendors
npm run build:packages
```

#### Build Output Structure

```
build/
├── vendors/              # React, ReactDOM, and other vendor libraries
│   ├── react.min.js
│   ├── react-dom.min.js
│   └── react-jsx-runtime.min.js
├── gutenberg/           # WordPress packages (dynamically built)
│   ├── element/
│   │   ├── index.min.js
│   │   └── index.min.asset.php
│   └── escape-html/
│       ├── index.min.js
│       └── index.min.asset.php
├── editor/              # Custom @blockeditor packages
│   ├── index.min.js
│   └── index.min.asset.php
├── block_editor.js      # Main entry bundle
└── block_editor.css     # Extracted styles
```

#### WordPress Package Integration

The build system automatically:

1. **Detects WordPress Packages**: Scans `package.json` dependencies for `@wordpress/*` packages
2. **Builds Packages**: Compiles each package with minification and source maps
3. **Extracts Dependencies**: Dynamically reads each package's `package.json` to determine dependencies
4. **Generates Asset Files**: Creates `.asset.php` files with dependency arrays
5. **Updates Libraries**: Auto-generates `block_editor.libraries.yml` with proper dependency chains

To add a new WordPress package:
```bash
npm install @wordpress/package-name
npm run build
```

The package will automatically be:
- Built to `build/gutenberg/package-name/`
- Added to `block_editor.libraries.yml`
- Available as `block_editor/wordpress_package_name`

#### Custom Package Development

Custom packages in `packages/` follow this structure:
```
packages/
└── editor/
    ├── package.json
    ├── src/
    └── build-module/
```

They are automatically:
- Built with Babel and webpack
- Generate asset files with dependencies
- Added to the Drupal libraries

#### Styling

- **SCSS/SASS Support**: Import `.scss` files in JavaScript
- **PostCSS Processing**: Automatic autoprefixing
- **CSS Extraction**: Styles are extracted to `build/block_editor.css`

Example:
```javascript
import './style.scss';
```

#### Library Auto-Generation

The `block_editor.libraries.yml` file is **automatically generated** during build:

- **Vendor Libraries**: React, ReactDOM, react-jsx-runtime
- **WordPress Packages**: All installed `@wordpress/*` packages with proper dependencies
- **Custom Packages**: Packages from `packages/` directory
- **Main Libraries**: Main entry points with all dependencies

**Do not edit `block_editor.libraries.yml` manually** - it will be overwritten on build.

### Adding Dependencies

#### JavaScript Dependencies

```bash
# Add a WordPress package
npm install @wordpress/block-editor

# Add a regular dependency
npm install some-package

# Rebuild to update libraries
npm run build
```

#### External Dependencies

External packages (like Drupal or custom globals) are configured in `webpack.config.js`:

```javascript
externals: {
  Drupal: 'Drupal',
  '@blockeditor/editor': 'be.editor',
}
```

### Webpack Configuration

The build system uses multiple webpack configurations:

1. **packages.js**: Builds custom packages from `packages/` directory
2. **vendors.js**: Copies vendor libraries (React, ReactDOM)
3. **wordpress-packages.js**: Dynamically builds WordPress packages
4. **webpack.config.js**: Main entry point with SCSS/CSS processing

Key features:
- Dynamic dependency extraction from package.json files
- Automatic library generation
- Production (minified) and development (source maps) builds
- CSS extraction with PostCSS and autoprefixer

## CONFIGURATION

1. Enable the module in Drupal admin
2. Configure permissions for block editing
3. Customize block settings as needed

## File Structure

```
block_editor/
├── src/                         # PHP source code
│   ├── Form/                   # Drupal forms
│   └── Controller/             # Controllers
├── js/                         # JavaScript source
│   ├── index.js               # Main entry point
│   └── style.scss             # Main styles
├── packages/                   # Custom packages
│   └── editor/                # @blockeditor/editor package
├── tools/
│   └── webpack/               # Webpack configuration
│       ├── packages.js        # Custom package builder
│       ├── vendors.js         # Vendor copier
│       ├── wordpress-packages.js  # WP package builder
│       └── generate-libraries.js  # Library generator
├── build/                      # Build output (auto-generated)
├── block_editor.libraries.yml  # Drupal libraries (auto-generated)
├── webpack.config.js          # Main webpack config
└── package.json               # JavaScript dependencies
```

## MAINTAINERS

Current maintainers:

- Frontkom Team - https://www.frontkom.com
