<?php

namespace Drupal\block_editor\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\block_editor\Service\EntityManager;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Route;

/**
 * Controls access to canonical routes when Block Editor is enabled.
 *
 * For entity types where canonical IS the edit form (like block_content),
 * this access check marks the request so an event subscriber can redirect
 * to the Block Editor edit form when Block Editor is enabled.
 */
class BlockEditorCanonicalAccessCheck implements AccessInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Block Editor entity manager.
   *
   * @var \Drupal\block_editor\Service\EntityManager
   */
  protected EntityManager $entityManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * Constructs a new BlockEditorCanonicalAccessCheck.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\block_editor\Service\EntityManager $entity_manager
   *   The Block Editor entity manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityManager $entity_manager, RequestStack $request_stack) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityManager = $entity_manager;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public function access(Route $route, RouteMatchInterface $route_match) {
    $entity = $this->extractContentEntity($route_match);

    if (!$entity) {
      return AccessResult::allowed();
    }

    $bundle_entity = $this->loadBundleEntity($entity);
    if (!$bundle_entity) {
      return AccessResult::allowed();
    }

    // If Block Editor is enabled, mark this request for redirection.
    if ($this->entityManager->isBlockEditorEnabledForEntity($bundle_entity)) {
      // Store entity info in the request for the event subscriber.
      $request = $this->requestStack->getCurrentRequest();
      $request->attributes->set('_block_editor_redirect_entity', $entity);

      // Allow access but mark for redirect by event subscriber.
      return AccessResult::allowed()
        ->addCacheableDependency($bundle_entity)
        ->setCacheMaxAge(0);
    }

    // If Block Editor is not enabled, allow normal access.
    return AccessResult::allowed()->addCacheableDependency($bundle_entity);
  }

  /**
   * Extracts the first content entity from the route parameters.
   */
  protected function extractContentEntity(RouteMatchInterface $route_match): ?ContentEntityInterface {
    foreach ($route_match->getParameters() as $parameter) {
      if ($parameter instanceof ContentEntityInterface) {
        return $parameter;
      }
    }

    return NULL;
  }

  /**
   * Loads the bundle config entity for a content entity.
   */
  protected function loadBundleEntity(ContentEntityInterface $entity): ?object {
    $entity_type = $entity->getEntityType();
    $bundle_entity_type_id = $entity_type->getBundleEntityType();

    if (!$bundle_entity_type_id) {
      return NULL;
    }

    $bundle_storage = $this->entityTypeManager->getStorage($bundle_entity_type_id);
    return $bundle_storage->load($entity->bundle());
  }

}
