<?php

namespace Drupal\block_editor\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\block_editor\Service\EntityManager;
use Symfony\Component\Routing\Route;

/**
 * Access check for Block Editor settings routes.
 *
 * Grants access only when Block Editor is enabled for the specific entity.
 */
class BlockEditorSettingsAccessCheck implements AccessInterface {

  /**
   * The Block Editor entity manager service.
   *
   * @var \Drupal\block_editor\Service\EntityManager
   */
  protected $entityManager;

  /**
   * Constructs a BlockEditorSettingsAccessCheck object.
   *
   * @param \Drupal\block_editor\Service\EntityManager $entity_manager
   *   The Block Editor entity manager service.
   */
  public function __construct(EntityManager $entity_manager) {
    $this->entityManager = $entity_manager;
  }

  /**
   * Checks access to Block Editor settings forms.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The parametrized route.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Route $route, RouteMatchInterface $route_match) {
    // Get the entity parameter from the route.
    $parameters = $route_match->getParameters();

    // Find the config entity in the route parameters.
    // The parameter name is the bundle entity type ID (e.g., node_type).
    foreach ($parameters as $parameter_name => $parameter) {
      // Skip the _raw_variables parameter.
      if ($parameter_name === '_raw_variables') {
        continue;
      }

      if ($parameter instanceof ConfigEntityInterface) {
        // Check if this entity type SUPPORTS Block Editor (has schema).
        if (!$this->entityManager->supportsBlockEditorSettings($parameter)) {
          return AccessResult::forbidden()->addCacheableDependency($parameter);
        }

        // Check if Block Editor is ENABLED for this specific entity.
        // Only show the settings tab when it's actually enabled.
        $enabled = $parameter->getThirdPartySetting('block_editor', 'enabled', FALSE);
        if (!$enabled) {
          return AccessResult::forbidden()
            ->addCacheableDependency($parameter)
            ->addCacheTags($parameter->getCacheTags());
        }

        return AccessResult::allowed()
          ->addCacheableDependency($parameter)
          ->addCacheTags($parameter->getCacheTags());
      }
    }

    // No config entity found in route parameters.
    return AccessResult::forbidden();
  }

}
