<?php

namespace Drupal\block_editor\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\block_editor\Service\EntityManager;
use Symfony\Component\Routing\Route;

/**
 * Determines access to entity edit form.
 *
 * When an entity type has Block Editor enabled, entity's edit form will be
 * "replaced" with Block Editor editor. This access check ensures that the
 * access to the entity edit form is forbidden.
 */
class EntityEditAccessCheck implements AccessInterface {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The Block Editor entity manager service.
   *
   * @var \Drupal\block_editor\Service\EntityManager
   */
  protected $entityManager;

  /**
   * Constructs an EntityEditAccessCheck object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\block_editor\Service\EntityManager $entity_manager
   *   The Block Editor entity manager service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityManager $entity_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityManager = $entity_manager;
  }

  /**
   * Checks access to entity edit forms in Block Editor context.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The parametrized route.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Route $route, RouteMatchInterface $route_match) {
    $requirement = $route->getRequirement('_entity_access');
    [$entity_type] = explode('.', $requirement);
    $parameters = $route_match->getParameters();

    if ($parameters->has($entity_type)) {
      $entity = $parameters->get($entity_type);

      if ($entity instanceof ContentEntityInterface) {
        $bundle_entity_type_id = $entity->getEntityType()->getBundleEntityType();

        if ($bundle_entity_type_id) {
          $bundle_storage = $this->entityTypeManager->getStorage($bundle_entity_type_id);
          $bundle = $bundle_storage ? $bundle_storage->load($entity->bundle()) : NULL;

          if ($bundle && $this->entityManager->isBlockEditorEnabledForEntity($bundle)) {
            return AccessResult::forbidden()->addCacheableDependency($bundle);
          }
        }
      }
    }

    return AccessResult::allowed();
  }

}
