<?php

namespace Drupal\block_editor\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\block_editor\Service\EntityManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for Block Editor-related routes.
 */
class BlockEditorController extends ControllerBase implements ContainerInjectionInterface {
  use StringTranslationTrait;

  /**
   * The entity type bundle info.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityTypeBundleInfo;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The entity manager.
   *
   * @var \Drupal\block_editor\Service\EntityManager
   */
  protected $entityManager;

  /**
   * Constructs a new BlockEditorController.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation.
   * @param \Drupal\Core\Entity\EntityFormBuilderInterface $entity_form_builder
   *   The entity form builder.
   * @param \Drupal\block_editor\Service\EntityManager $entity_manager
   *   The entity manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfoInterface $entity_type_bundle_info, EntityRepositoryInterface $entity_repository, TranslationInterface $string_translation, EntityFormBuilderInterface $entity_form_builder, EntityManager $entity_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->entityRepository = $entity_repository;
    $this->stringTranslation = $string_translation;
    $this->entityFormBuilder = $entity_form_builder;
    $this->entityManager = $entity_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('entity_type.manager'),
          $container->get('entity_type.bundle.info'),
          $container->get('entity.repository'),
          $container->get('string_translation'),
          $container->get('entity.form_builder'),
          $container->get('block_editor.entity_manager'),
      );
  }

  /**
   * Provides a generic add title callback.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param \Drupal\Core\Entity\EntityInterface $_entity
   *   (optional) An entity, passed in directly from the request attributes.
   *
   * @return string|null
   *   The title for the entity edit page, if an entity was found.
   */
  public function addTitle(RouteMatchInterface $route_match, ?EntityInterface $_entity = NULL) {
    if ($entity = $this->doGetEntity($route_match, $_entity)) {
      return $this->t('Create %label', ['%label' => $entity->label()]);
    }
  }

  /**
   * Provides a generic edit title callback.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param \Drupal\Core\Entity\EntityInterface $_entity
   *   (optional) An entity, passed in directly from the request attributes.
   *
   * @return string|null
   *   The title for the entity edit page, if an entity was found.
   */
  public function editTitle(RouteMatchInterface $route_match, ?EntityInterface $_entity = NULL) {
    if ($entity = $this->doGetEntity($route_match, $_entity)) {
      return $this->t('Edit %label', ['%label' => $entity->label()]);
    }
  }

  /**
   * Provides the title for Block Editor settings pages.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The translated title.
   */
  public function settingsTitle() {
    return $this->t('Manage Block Editor');
  }

  /**
   * Provides the entity add form.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param \Drupal\Core\Entity\EntityInterface $_entity
   *   (optional) An entity, passed in directly from the request attributes.
   *
   * @return array
   *   The entity add form.
   */
  public function addForm(RouteMatchInterface $route_match, ?EntityInterface $_entity = NULL) {
    $bundle_entity = $this->doGetEntity($route_match, $_entity);

    if (!$bundle_entity) {
      return [];
    }

    // For add forms, the entity is the bundle entity (e.g., node_type).
    // We need to create a new content entity using this bundle.
    $bundle_entity_type = $bundle_entity->getEntityType();
    $content_entity_type_id = $bundle_entity_type->getBundleOf();

    if (!$content_entity_type_id) {
      // Fallback to default form if no bundle relationship.
      return $this->entityFormBuilder->getForm($bundle_entity);
    }

    // Create new content entity with the bundle.
    $content_entity = $this->entityTypeManager
      ->getStorage($content_entity_type_id)
      ->create(
              [
                $bundle_entity_type->getKey('id') => $bundle_entity->id(),
                'type' => $bundle_entity->id(),
              ]
          );

    // Check if Block Editor is enabled for this bundle.
    if ($this->entityManager->isBlockEditorEnabledForEntity($bundle_entity)) {
      // Use Block Editor form.
      return $this->entityFormBuilder->getForm($content_entity, 'block_editor');
    }

    // Use default form.
    return $this->entityFormBuilder->getForm($content_entity);
  }

  /**
   * Determines the entity for the route.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match.
   * @param \Drupal\Core\Entity\EntityInterface $_entity
   *   (optional) The entity object.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The entity object or NULL.
   */
  protected function doGetEntity(RouteMatchInterface $route_match, ?EntityInterface $_entity = NULL) {
    if ($_entity) {
      return $_entity;
    }

    // Check all route parameters to find an entity.
    $parameters = $route_match->getParameters();
    foreach ($parameters as $parameter) {
      if ($parameter instanceof EntityInterface) {
        return $parameter;
      }
    }

    return NULL;
  }

}
