<?php

namespace Drupal\block_editor\EventSubscriber;

use Drupal\Core\Url;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Redirects canonical routes to Block Editor when enabled.
 *
 * For entity types where the canonical route IS the edit form
 * (like block_content), this subscriber redirects to the Block Editor
 * edit form when Block Editor is enabled.
 */
class BlockEditorCanonicalRedirectSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Run after routing (priority 31) but before controller resolution.
    $events[KernelEvents::REQUEST][] = ['onRequest', 30];
    return $events;
  }

  /**
   * Redirects to Block Editor edit form if marked by access check.
   */
  public function onRequest(RequestEvent $event): void {
    $request = $event->getRequest();

    // Check if the access check marked this request for redirection.
    $entity = $request->attributes->get('_block_editor_redirect_entity');
    if (!$entity) {
      return;
    }

    $entity_type_id = $entity->getEntityTypeId();
    $url = Url::fromRoute(
      'block_editor.entity.' . $entity_type_id . '.edit_form',
      [$entity_type_id => $entity->id()]
    );

    $response = new RedirectResponse($url->toString());
    $event->setResponse($response);
  }

}
