<?php

namespace Drupal\block_editor\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\block_editor\Service\EntityManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for Block Editor edit forms.
 */
class EntityEditForm extends ContentEntityForm {
  /**
   * The entity being used by this form.
   *
   * @var \Drupal\Core\Entity\ContentEntityInterface|\Drupal\Core\Entity\RevisionLogInterface
   */
  protected $entity;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityTypeBundleInfo;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The entity repository service.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The entity manager service.
   *
   * @var \Drupal\block_editor\Service\EntityManager
   */
  protected $entityManager;

  /**
   * The entity form builder service.
   *
   * @var \Drupal\Core\Entity\EntityFormBuilderInterface
   */
  protected $entityFormBuilder;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a ContentEntityForm object.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Entity\EntityFormBuilderInterface $entity_form_builder
   *   The entity form builder service.
   * @param \Drupal\block_editor\Service\EntityManager $entity_manager
   *   The entity manager service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct(EntityRepositoryInterface $entity_repository, EntityTypeBundleInfoInterface $entity_type_bundle_info, TimeInterface $time, EntityFormBuilderInterface $entity_form_builder, EntityManager $entity_manager, ModuleHandlerInterface $module_handler) {
    $this->entityRepository = $entity_repository;
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->time = $time;
    $this->entityFormBuilder = $entity_form_builder;
    $this->entityManager = $entity_manager;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity.repository'),
      $container->get('entity_type.bundle.info'),
      $container->get('datetime.time'),
      $container->get('entity.form_builder'),
      $container->get('block_editor.entity_manager'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $entity = $this->getEntity();

    // Get the bundle entity to check if Block Editor is enabled.
    $bundle = $this->entityTypeManager
      ->getStorage($entity->getEntityType()->getBundleEntityType())
      ->load($entity->bundle());

    // If Block Editor is not enabled, use default form behavior.
    // This handles both add and edit forms.
    if (!$bundle || !$this->entityManager->isBlockEditorEnabledForEntity($bundle)) {
      // Set the operation to default to ensure proper form building.
      $this->operation = 'default';
      // Use parent form building which will render the standard entity form.
      return parent::form($form, $form_state);
    }

    // Block Editor is enabled, so build our custom form.
    $form = parent::form($form, $form_state);

    // ISOLATION: Mark form as Block Editor to prevent standard alterations.
    $form['#block_editor_form'] = TRUE;
    $form['#block_editor_entity_type'] = $entity->getEntityTypeId();
    $form['#block_editor_bundle'] = $entity->bundle();

    // Set custom theme for Block Editor forms.
    if (!isset($form['#theme'])) {
      $form['#theme'] = [];
    }
    // Add block-editor specific theme suggestion.
    // @todo Uncomment when theme is ready.
    // $form['#theme'][] = 'form__block_editor_form';
    // CUSTOM HOOKS: Invoke Block Editor specific form alter hooks.
    $this->invokeBlockEditorFormAlterHooks($form, $form_state, $entity);

    $form['#attributes']['class'][] = 'block-editor-form';
    $form['#attributes']['data-block-editor'] = 'true';
    $form['#attributes']['data-entity-type'] = $entity->getEntityTypeId();
    $form['#attributes']['data-bundle'] = $entity->bundle();

    if (!isset($form['meta'])) {
      $form['meta'] = [
        '#group' => 'advanced',
        '#weight' => -10,
        '#title' => $this->t('Status'),
        '#attributes' => ['class' => ['entity-meta__header']],
        '#tree' => TRUE,
      ];
    }

    $form['advanced']['#type'] = 'container';
    $form['advanced']['#accordion'] = TRUE;
    $form['meta']['#type'] = 'container';
    $form['meta']['#access'] = TRUE;

    $form['revision_information']['#type'] = 'container';
    $form['revision_information']['#group'] = 'meta';
    $form['revision_information']['#weight'] = '10';
    $form['revision_information']['#attributes']['class'][] = 'entity-meta__revision';

    $form['status']['#group'] = 'meta';
    $form['status']['#weight'] = 0;
    $form['status']['#attributes']['class'][] = 'entity-meta__status';

    $form['promote']['#group'] = 'meta';
    $form['promote']['#attributes']['class'][] = 'entity-meta__promote';

    $form['sticky']['#group'] = 'meta';
    $form['sticky']['#attributes']['class'][] = 'entity-meta__promote';

    if (!isset($form['authoring']) && (isset($form['uid']) || isset($form['created']))) {
      $form['authoring'] = [
        '#group' => 'advanced',
        '#weight' => 100,
        '#type' => 'details',
        '#accordion' => TRUE,
        '#title' => $this->t('Authoring'),
        '#attributes' => ['class' => ['entity-meta__authoring']],
        '#tree' => TRUE,
      ];

      $form['uid']['#group'] = 'authoring';
      $form['created']['#group'] = 'authoring';
    }

    // $editor_field = $this->entityManager->getGutenbergField($entity);
    // $field_name = $editor_field->getName();
    // $form[$field_name]['widget'][0]['#format'] = 'gutenberg';
    // $form[$field_name]['#attributes']['class'][] = 'field--gutenberg';
    // Hide the field label.
    // $form[$field_name]['widget'][0]['#title_display'] = 'hidden';
    // Call after build for the field.
    // $form[$field_name]['widget']['#after_build'][] =
    // 'gutenberg_form_node_form_text_field_after_build';
    // Add Block Editor interface placeholder.
    $form['block_editor'] = [
      '#type' => 'markup',
      '#markup' => '<div id="block-editor-root" class="block-editor-interface"><p>' . $this->t('Block Editor interface will be rendered here.') . '</p></div>',
      '#weight' => -10,
    ];

    // Attach Block Editor libraries and settings (when implemented).
    /* $form['#attached']['library'][] = 'block_editor/editor'; */
    $form['#attached']['drupalSettings']['blockEditor'] = [
      'entityType' => $entity->getEntityTypeId(),
      'bundle' => $entity->bundle(),
      'entityId' => $entity->id(),
    ];

    $form['#attached']['library'][] = 'block_editor/block_editor_form';

    return $form;
  }

  /**
   * Invokes Block Editor specific form alter hooks.
   *
   * This provides dedicated hooks for Block Editor forms, separate from
   * standard entity form alterations.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity being edited.
   */
  protected function invokeBlockEditorFormAlterHooks(array &$form, FormStateInterface $form_state, $entity): void {
    $entity_type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();

    // Get module handler to invoke hooks.
    $module_handler = $this->moduleHandler;

    // 1. General Block Editor form alter hook.
    $module_handler->alter('block_editor_form', $form, $form_state, $entity);

    // 2. Entity type specific hooks (e.g., node_block_editor_form_alter).
    $hook = $entity_type_id . '_block_editor_form';
    $module_handler->alter($hook, $form, $form_state, $entity);

    // 3. Bundle specific hooks (e.g., node_article_block_editor_form_alter).
    if ($bundle !== $entity_type_id) {
      $bundle_hook = $entity_type_id . '_' . $bundle . '_block_editor_form';
      $module_handler->alter($bundle_hook, $form, $form_state, $entity);
    }

    // 4. Form ID specific hook.
    $form_id = $this->getFormId();
    $form_id_hook = $form_id . '_block_editor';
    $module_handler->alter($form_id_hook, $form, $form_state, $entity);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity = $this->getEntity();
    $status = parent::save($form, $form_state);

    $this->messenger()->addStatus(
      $this->t(
        '@type %title has been saved.',
        [
          '@type' => $entity->getEntityType()->getLabel(),
          '%title' => $entity->label(),
        ]
      )
    );

    // Redirect to the entity's canonical page.
    $form_state->setRedirect('entity.' . $entity->getEntityTypeId() . '.canonical', [$entity->getEntityTypeId() => $entity->id()]);

    return $status;
  }

}
