<?php

namespace Drupal\block_editor\Form;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for managing Block Editor settings for entity types.
 */
class EntityTypeManageForm extends FormBase implements ContainerInjectionInterface {
  use StringTranslationTrait;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs a new EntityTypeManageForm.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   */
  public function __construct(RouteMatchInterface $route_match) {
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('current_route_match')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'block_editor_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Get the config entity from route parameters dynamically.
    $entity = NULL;
    $route_match = $this->routeMatch;
    $parameters = $route_match->getParameters();

    // Find the config entity in the route parameters.
    foreach ($parameters as $parameter_name => $parameter) {
      // Skip special parameters.
      if ($parameter_name === '_raw_variables' || $parameter_name === '_route' || $parameter_name === '_route_object') {
        continue;
      }

      if ($parameter instanceof ConfigEntityInterface) {
        $entity = $parameter;
        break;
      }
    }

    if (!$entity) {
      $form['error'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('No entity found in route parameters.') . '</p>',
      ];
      return $form;
    }

    // The access check already ensures Block Editor is enabled,
    // so we can proceed directly to the form fields.
    $form['entity'] = [
      '#type' => 'value',
      '#value' => $entity,
    ];

    $form['template'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Template'),
      '#description' => $this->t(
          'Define a template for this @entity_type. Use JSON format to define the block structure.', [
            '@entity_type' => strtolower($entity->getEntityType()->getLabel()),
          ]
      ),
      '#default_value' => $entity->getThirdPartySetting('block_editor', 'template', ''),
      '#rows' => 8,
    ];

    $form['template_lock'] = [
      '#type' => 'select',
      '#title' => $this->t('Template lock'),
      '#description' => $this->t('Restrict which parts of the template users can modify.'),
      '#options' => [
        '' => $this->t('- None -'),
        'all' => $this->t('All - Users cannot modify anything'),
        'insert' => $this->t('Insert - Users cannot insert or remove blocks'),
        'contentOnly' => $this->t('Content Only - Users can only edit block content'),
      ],
      '#default_value' => $entity->getThirdPartySetting('block_editor', 'template_lock', ''),
    ];

    $form['allowed_blocks'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed blocks'),
      '#description' => $this->t('Specify which blocks are allowed. One block name per line. Leave empty to allow all blocks.'),
      '#default_value' => implode("\n", $entity->getThirdPartySetting('block_editor', 'allowed_blocks', [])),
      '#rows' => 5,
    ];

    $form['allowed_drupal_blocks'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed Drupal blocks'),
      '#description' => $this->t('Specify which Drupal blocks are allowed. One block ID per line.'),
      '#default_value' => implode("\n", $entity->getThirdPartySetting('block_editor', 'allowed_drupal_blocks', [])),
      '#rows' => 5,
    ];

    $form['allowed_image_styles'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed image styles'),
      '#description' => $this->t('Specify which image styles are allowed. One style name per line.'),
      '#default_value' => implode("\n", $entity->getThirdPartySetting('block_editor', 'allowed_image_styles', [])),
      '#rows' => 5,
    ];

    $form['allowed_content_block_types'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed content block types'),
      '#description' => $this->t('Specify which content block types are allowed. One type per line.'),
      '#default_value' => implode("\n", $entity->getThirdPartySetting('block_editor', 'allowed_content_block_types', [])),
      '#rows' => 5,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save settings'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $entity = $form_state->getValue('entity');

    if (!$entity instanceof ConfigEntityInterface) {
      return;
    }

    // Save template settings.
    $template = trim($form_state->getValue('template', ''));
    if (!empty($template)) {
      $entity->setThirdPartySetting('block_editor', 'template', $template);
    }
    else {
      $entity->unsetThirdPartySetting('block_editor', 'template');
    }

    // Save template lock.
    $template_lock = $form_state->getValue('template_lock', '');
    if (!empty($template_lock)) {
      $entity->setThirdPartySetting('block_editor', 'template_lock', $template_lock);
    }
    else {
      $entity->unsetThirdPartySetting('block_editor', 'template_lock');
    }

    // Save allowed blocks.
    $allowed_blocks = array_filter(array_map('trim', explode("\n", $form_state->getValue('allowed_blocks', ''))));
    if (!empty($allowed_blocks)) {
      $entity->setThirdPartySetting('block_editor', 'allowed_blocks', $allowed_blocks);
    }
    else {
      $entity->unsetThirdPartySetting('block_editor', 'allowed_blocks');
    }

    // Save allowed Drupal blocks.
    $allowed_drupal_blocks = array_filter(array_map('trim', explode("\n", $form_state->getValue('allowed_drupal_blocks', ''))));
    if (!empty($allowed_drupal_blocks)) {
      $entity->setThirdPartySetting('block_editor', 'allowed_drupal_blocks', $allowed_drupal_blocks);
    }
    else {
      $entity->unsetThirdPartySetting('block_editor', 'allowed_drupal_blocks');
    }

    // Save allowed image styles.
    $allowed_image_styles = array_filter(array_map('trim', explode("\n", $form_state->getValue('allowed_image_styles', ''))));
    if (!empty($allowed_image_styles)) {
      $entity->setThirdPartySetting('block_editor', 'allowed_image_styles', $allowed_image_styles);
    }
    else {
      $entity->unsetThirdPartySetting('block_editor', 'allowed_image_styles');
    }

    // Save allowed content block types.
    $allowed_content_block_types = array_filter(array_map('trim', explode("\n", $form_state->getValue('allowed_content_block_types', ''))));
    if (!empty($allowed_content_block_types)) {
      $entity->setThirdPartySetting('block_editor', 'allowed_content_block_types', $allowed_content_block_types);
    }
    else {
      $entity->unsetThirdPartySetting('block_editor', 'allowed_content_block_types');
    }

    // Save the entity.
    $entity->save();

    $this->messenger()->addMessage($this->t('Block Editor settings have been saved.'));
  }

}
