<?php

namespace Drupal\block_editor\Hook;

use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\block_editor\Service\EntityManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Hook implementations for forms.
 */
class FormHooks implements ContainerInjectionInterface {
  use StringTranslationTrait;

  /**
   * The Block Editor entity manager.
   *
   * @var \Drupal\block_editor\Service\EntityManager
   */
  protected $entityManager;

  /**
   * Constructs a new FormHooks.
   *
   * @param \Drupal\block_editor\Service\EntityManager $entity_manager
   *   The Block Editor entity manager.
   */
  public function __construct(EntityManager $entity_manager) {
    $this->entityManager = $entity_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('block_editor.entity_manager')
      );
  }

  /**
   * Implements hook_form_alter().
   *
   * Adds Block Editor settings to entity type edit forms that support it.
   */
  #[Hook('form_alter')]
  public function formAlter(array &$form, FormStateInterface $form_state, $form_id) {
    $form_object = $form_state->getFormObject();

    // Only process entity forms.
    if (!$form_object instanceof EntityFormInterface) {
      return;
    }

    $entity = $form_object->getEntity();

    if ($entity instanceof ConfigEntityInterface && $this->entityManager->supportsBlockEditorSettings($entity)) {
      // Add Block Editor settings to the form.
      $this->addBlockEditorSettings($form, $entity);
    }

    // Only set Block Editor theme if Block Editor is enabled for this form.
    if (isset($form['#block_editor_form']) && $form['#block_editor_form'] === TRUE) {
      $form['#after_build'][] = function (array $form, FormStateInterface $form_state) {
        $form['#theme'] = ['block_editor__form'];
        return $form;
      };
    }
  }

  /**
   * Adds Block Editor settings to the entity form.
   *
   * @param array &$form
   *   The form array.
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $entity
   *   The config entity.
   */
  protected function addBlockEditorSettings(array &$form, $entity) {
    // Add the enabled checkbox.
    $form['block_editor_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Block Editor'),
      '#description' => $this->t(
          'Enable the Block Editor for this @entity_type. Additional settings will be available after saving.', [
            '@entity_type' => strtolower($entity->getEntityType()->getLabel()),
          ]
      ),
      '#default_value' => $entity->getThirdPartySetting('block_editor', 'enabled', FALSE),
      '#weight' => 0.002,
    ];

    // Use entity_builder to handle the form submission.
    $form['#entity_builders'][] = [$this, 'blockEditorEntityBuilder'];
  }

  /**
   * Entity builder for Block Editor settings.
   *
   * This method is called during entity building to update the entity with
   * form values before it's saved.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param \Drupal\Core\Config\Entity\ConfigEntityInterface $entity
   *   The entity being built.
   * @param array &$form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function blockEditorEntityBuilder($entity_type_id, $entity, array &$form, FormStateInterface $form_state) {
    if (!$entity instanceof ConfigEntityInterface) {
      return;
    }

    $enabled = $form_state->getValue('block_editor_enabled', FALSE);

    if ($enabled) {
      // Enable Block Editor.
      $entity->setThirdPartySetting('block_editor', 'enabled', TRUE);
    }
    else {
      // Remove all Block Editor settings if disabled.
      $entity->unsetThirdPartySetting('block_editor', 'enabled');
      $entity->unsetThirdPartySetting('block_editor', 'template');
      $entity->unsetThirdPartySetting('block_editor', 'template_lock');
      $entity->unsetThirdPartySetting('block_editor', 'allowed_blocks');
      $entity->unsetThirdPartySetting('block_editor', 'allowed_drupal_blocks');
      $entity->unsetThirdPartySetting('block_editor', 'allowed_image_styles');
      $entity->unsetThirdPartySetting('block_editor', 'allowed_content_block_types');
    }
  }

}
