<?php

namespace Drupal\block_editor\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Entity\ContentEntityType;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\block_editor\Service\EntityManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides dynamic local tasks for Block Editor entity forms.
 */
class BlockEditorLocalTask extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity manager.
   *
   * @var \Drupal\block_editor\Service\EntityManager
   */
  protected $entityManager;

  /**
   * Constructs a new BlockEditorLocalTask.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\block_editor\Service\EntityManager $entity_manager
   *   The entity manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityManager $entity_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityManager = $entity_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('block_editor.entity_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    $this->derivatives = [];

    $entity_types = $this->entityTypeManager->getDefinitions();

    foreach ($entity_types as $entity_type_id => $entity_type) {
      // Only process content entity types.
      if (!($entity_type instanceof ContentEntityType)) {
        continue;
      }

      // Check if the entity type has a bundle entity type.
      $config_entity_type_id = $entity_type->getBundleEntityType();
      if (empty($config_entity_type_id)) {
        continue;
      }

      $config_entity_type = $this->entityTypeManager
        ->getDefinition($config_entity_type_id);

      // Check if the entity type supports Block Editor.
      if (!$this->entityManager->entityTypeSupportsBlockEditor($config_entity_type)) {
        continue;
      }

      // Check if canonical and edit-form are the same for this entity type.
      $canonical_link = $entity_type->getLinkTemplate('canonical');
      $edit_form_link = $entity_type->getLinkTemplate('edit-form');
      $is_canonical_same_as_edit = ($canonical_link === $edit_form_link);

      // Create a local task for this entity type.
      // The tab will show on the entity view page and link to the Block
      // Editor edit form.
      // For entity types where canonical IS edit (like block_content),
      // use negative weight to ensure it appears first.
      // For other entity types (like node), use positive weight.
      $weight = $is_canonical_same_as_edit ? -10 : 5;

      $this->derivatives["block_editor.{$entity_type_id}.block_editor_edit"] = [
        'route_name' => "block_editor.entity.{$entity_type_id}.edit_form",
        'base_route' => "entity.{$entity_type_id}.canonical",
        'title' => $this->t('Edit'),
        'weight' => $weight,
      ] + $base_plugin_definition;
    }

    return $this->derivatives;
  }

}
