<?php

namespace Drupal\block_editor\Plugin\Derivative;

use Drupal\Component\Plugin\Derivative\DeriverBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Discovery\ContainerDeriverInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\block_editor\Service\EntityManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides dynamic local tasks for Block Editor settings tabs.
 *
 * This automatically creates "Manage Block Editor" tabs for all supported
 * bundle entity types (node_type, taxonomy_vocabulary, etc.).
 */
class BlockEditorSettingsLocalTask extends DeriverBase implements ContainerDeriverInterface {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity manager.
   *
   * @var \Drupal\block_editor\Service\EntityManager
   */
  protected $entityManager;

  /**
   * Constructs a new BlockEditorSettingsLocalTask.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\block_editor\Service\EntityManager $entity_manager
   *   The entity manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityManager $entity_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityManager = $entity_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $base_plugin_id) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('block_editor.entity_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getDerivativeDefinitions($base_plugin_definition) {
    $this->derivatives = [];

    // Get all entity types that support Block Editor.
    $supported_mappings = $this->entityManager->getSupportedEntityTypeMappings();

    foreach ($supported_mappings as $bundle_entity_type_id => $content_entity_type_id) {
      // Determine the base route for the local task.
      // Some entity types have special base routes.
      $base_route = $this->getBaseRoute($bundle_entity_type_id);

      // Create a local task for the settings page.
      // This will appear as a tab on the bundle entity edit form.
      $this->derivatives["block_editor.{$bundle_entity_type_id}.settings"] = [
        'route_name' => "block_editor.{$bundle_entity_type_id}.settings",
        'base_route' => $base_route,
        'title' => $this->t('Manage Block Editor'),
        'weight' => 10,
      ] + $base_plugin_definition;
    }

    return $this->derivatives;
  }

  /**
   * Gets the base route for a bundle entity type's local tasks.
   *
   * Some entity types use different base routes for their tabs.
   * For example, taxonomy_vocabulary uses overview_form instead of edit_form.
   *
   * @param string $bundle_entity_type_id
   *   The bundle entity type ID.
   *
   * @return string
   *   The base route name.
   */
  protected function getBaseRoute(string $bundle_entity_type_id): string {
    // Special cases where the base route is not edit_form.
    $special_cases = [
      'taxonomy_vocabulary' => 'entity.taxonomy_vocabulary.overview_form',
    ];

    if (isset($special_cases[$bundle_entity_type_id])) {
      return $special_cases[$bundle_entity_type_id];
    }

    // Default: use edit_form as base route.
    return "entity.{$bundle_entity_type_id}.edit_form";
  }

}
