<?php

namespace Drupal\block_editor\Service;

use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Service for discovering and managing Block Editor supported entity types.
 */
class EntityTypeDiscovery {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Cached supported entity types.
   *
   * @var array|null
   */
  protected $supportedEntityTypes;

  /**
   * Constructs an EntityTypeDiscovery object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   */
  public function __construct(ModuleHandlerInterface $module_handler) {
    $this->moduleHandler = $module_handler;
  }

  /**
   * Gets all supported entity types for Block Editor.
   *
   * @return array
   *   Array of entity type definitions keyed by bundle entity type ID.
   */
  public function getSupportedEntityTypes(): array {
    if ($this->supportedEntityTypes === NULL) {
      $this->supportedEntityTypes = [];

      // Allow modules to define supported entity types.
      $this->moduleHandler->invokeAllWith(
        'block_editor_supported_entity_types',
        function (callable $hook, string $module) {
          $entity_types = $hook();
          if (is_array($entity_types)) {
            $this->supportedEntityTypes = array_merge($this->supportedEntityTypes, $entity_types);
          }
        }
      );
    }

    return $this->supportedEntityTypes;
  }

  /**
   * Gets the definition for a specific entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return array|null
   *   The entity type definition or NULL if not supported.
   */
  public function getEntityTypeDefinition(string $entity_type_id): ?array {
    $supported_types = $this->getSupportedEntityTypes();
    return $supported_types[$entity_type_id] ?? NULL;
  }

  /**
   * Checks if an entity type is supported.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return bool
   *   TRUE if the entity type is supported.
   */
  public function isEntityTypeSupported(string $entity_type_id): bool {
    return array_key_exists($entity_type_id, $this->getSupportedEntityTypes());
  }

  /**
   * Gets all content entity type IDs that have Block Editor support.
   *
   * @return array
   *   Array of content entity type IDs.
   */
  public function getSupportedContentEntityTypes(): array {
    $content_types = [];
    foreach ($this->getSupportedEntityTypes() as $definition) {
      if (isset($definition['content_entity_type'])) {
        $content_types[] = $definition['content_entity_type'];
      }
    }
    return array_unique($content_types);
  }

  /**
   * Gets route information for generating dynamic routes.
   *
   * @return array
   *   Array of route definitions for supported entity types.
   */
  public function getRouteDefinitions(): array {
    $routes = [];
    foreach ($this->getSupportedEntityTypes() as $entity_type_id => $definition) {
      $routes["block_editor.{$entity_type_id}.settings"] = [
        'path' => str_replace('{bundle}', "{{$entity_type_id}}", $definition['route_path_pattern']),
        'entity_type_id' => $entity_type_id,
        'permission' => $definition['admin_permission'],
      ];
    }
    return $routes;
  }

  /**
   * Clears the cached supported entity types.
   */
  public function clearCache(): void {
    $this->supportedEntityTypes = NULL;
  }

}
