<?php

namespace Drupal\Tests\block_editor\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Test Block Editor settings on content types.
 *
 * @group block_editor
 */
class BlockEditorSettingsTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'claro';

  /**
   * {@inheritdoc}
   */
  protected $strictConfigSchema = TRUE;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block_editor',
    'node',
  ];

  /**
   * Tests the Block Editor checkbox and settings page access.
   *
   * This test verifies:
   * - The "Enable Block Editor" checkbox appears on content type forms.
   * - The checkbox is unchecked by default.
   * - The Block Editor settings page is not accessible when disabled.
   * - Enabling the checkbox saves the third-party setting correctly.
   * - The Block Editor settings page becomes accessible after enabling.
   */
  public function testEnableBlockEditorCheckboxAndTab() {
    // Create the page content type.
    $this->drupalCreateContentType(['type' => 'page', 'name' => 'Page']);

    $account = $this->drupalCreateUser([
      'administer content types',
    ]);
    // Login but catch the session check assertion that sometimes fails.
    $this->drupalLogin($account);

    // Ensure the node type has Block Editor disabled.
    /** @var \Drupal\Core\Config\ConfigEntityInterface $node_type */
    $node_type = \Drupal::entityTypeManager()->getStorage('node_type')->load('page');
    $node_type->setThirdPartySetting('block_editor', 'enabled', FALSE);
    $node_type->save();

    // Go to the page content type edit form.
    $this->drupalGet('admin/structure/types/manage/page');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Page');

    // Check that the Enable Block Editor checkbox exists.
    $this->assertSession()->fieldExists('block_editor_enabled');

    // Check that it's not checked by default.
    $field = $this->getSession()->getPage()->find('css', 'input[name="block_editor_enabled"]');
    $this->assertNotNull($field);
    $this->assertFalse($field->isChecked());

    // Check that the Manage Block Editor settings page is not accessible.
    $this->drupalGet('admin/structure/types/manage/page/block-editor-settings');
    $this->assertSession()->statusCodeEquals(403);

    // Go back to the edit form and enable the Block Editor.
    $this->drupalGet('admin/structure/types/manage/page');
    $this->submitForm([
      'block_editor_enabled' => TRUE,
    ], 'Save');
    $this->assertSession()->pageTextContains('The content type Page has been updated.');

    // After saving, verify the setting was saved by reloading the entity.
    \Drupal::entityTypeManager()->getStorage('node_type')->resetCache(['page']);
    /** @var \Drupal\Core\Config\ConfigEntityInterface $node_type */
    $node_type = \Drupal::entityTypeManager()->getStorage('node_type')->load('page');
    $enabled = $node_type->getThirdPartySetting('block_editor', 'enabled', FALSE);
    $this->assertTrue($enabled, 'Block Editor should be enabled after form submission');

    // Rebuild router and local task caches to ensure routes are available.
    \Drupal::service('router.builder')->rebuild();

    // Verify we can access the Block Editor settings page.
    $this->drupalGet('admin/structure/types/manage/page/block-editor-settings');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Manage Block Editor');
  }

  /**
   * Tests that Block Editor form is shown when editing content.
   *
   * This test verifies:
   * - When Block Editor is disabled, the standard edit form is shown.
   * - When Block Editor is enabled, the Block Editor form is shown.
   * - The Block Editor form has Block Editor-specific elements.
   */
  public function testBlockEditorFormDisplay() {
    // Create the article content type.
    $this->drupalCreateContentType(['type' => 'article', 'name' => 'Article']);

    // Create a user with permissions to create and edit content.
    $account = $this->drupalCreateUser([
      'administer content types',
      'create article content',
      'edit any article content',
    ]);
    $this->drupalLogin($account);

    // Create a test article node.
    $node = $this->drupalCreateNode([
      'type' => 'article',
      'title' => 'Test Article',
      'body' => [
        'value' => 'Test content',
        'format' => 'plain_text',
      ],
    ]);

    // First, ensure Block Editor is disabled.
    /** @var \Drupal\Core\Config\ConfigEntityInterface $node_type */
    $node_type = \Drupal::entityTypeManager()->getStorage('node_type')->load('article');
    $node_type->setThirdPartySetting('block_editor', 'enabled', FALSE);
    $node_type->save();

    // Visit the edit form - should show standard Drupal form.
    $this->drupalGet('node/' . $node->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Edit Article');

    // Standard form should have the body field.
    $this->assertSession()->fieldExists('body[0][value]');

    // Should NOT have Block Editor elements.
    $this->assertSession()->elementNotExists('css', '#editor');
    $this->assertSession()->elementNotExists('css', '.block-editor');

    // Now enable Block Editor.
    $node_type->setThirdPartySetting('block_editor', 'enabled', TRUE);
    $node_type->save();

    // Rebuild router to ensure Block Editor routes are available.
    \Drupal::service('router.builder')->rebuild();

    // Visit the edit form again - should redirect to Block Editor form.
    $this->drupalGet('node/' . $node->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);

    // Should be redirected to block-editor route.
    $this->assertSession()->addressEquals('node/' . $node->id() . '/block-editor');

    // Block Editor form should have Block Editor-specific elements.
    // The exact elements depend on the Block Editor implementation,
    // but we can check for the form itself.
    $this->assertSession()->elementExists('css', 'form.block-editor-form, form[data-block-editor-form], #block-editor-form, form');

    // Verify the form has the special marker from Block Editor.
    $page_content = $this->getSession()->getPage()->getContent();
    $this->assertStringContainsString('block_editor', $page_content, 'Page should contain block_editor indicators');
  }

}
