<?php

namespace Drupal\block_purge\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\block\Entity\Block;
use Drupal\Core\Cache\Cache;

/**
 * Provides a form with checkboxes to select and clear block caches.
 */
class BlockPurgeForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'block_purge_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $active_blocks = $this->getActiveBlocks();

    if (!empty($active_blocks)) {
      $options = [];

      foreach ($active_blocks as $block_id => $block) {
        $label = $block->label();
        $plugin = $block->getPluginId();
        $region = $block->getRegion();
        $options[$block_id] = "$label (Plugin: $plugin, Region: $region)";
      }

      $form['block_cache_selection'] = [
        '#type' => 'details',
        '#title' => $this->t('Select Active Blocks'),
        '#open' => FALSE,
      ];

      $form['block_cache_selection']['block_selection'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Blocks'),
        '#options' => $options,
      ];
    }
    else {
      $form['no_blocks'] = [
        '#markup' => $this->t('No active blocks found.'),
      ];
    }

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Clear Selected Block Caches'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * Gets all active (enabled) blocks.
   *
   * @return \Drupal\block\Entity\Block[]
   *   An array of active block entities.
   */
  protected function getActiveBlocks() {
    $block_storage = \Drupal::entityTypeManager()->getStorage('block');
    $blocks = $block_storage->loadMultiple();

    return array_filter($blocks, function ($block) {
      return $block->status();
    });
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $selected_blocks = array_filter($form_state->getValue('block_selection'));

    if (empty($selected_blocks)) {
      $this->messenger()->addWarning($this->t('No blocks were selected. Please select at least one block.'));
      return;
    }

    foreach ($selected_blocks as $block_id) {
      if ($block = Block::load($block_id)) {
        $uuid = $block->uuid();

        // Invalidate block view cache.
        Cache::invalidateTags(["block_view:$uuid"]);

        // Invalidate render cache for the block.
        Cache::invalidateTags(["rendered"]);

        // Optional: Invalidate the block entity itself (not always needed).
        Cache::invalidateTags(["config:block.block.$block_id"]);
      }
    }

    $this->messenger()->addStatus($this->t('Cache cleared for the selected blocks. If your blocks still appear outdated, consider rebuilding the page or flushing render cache.'));
  }

}
