# Routing / Aliases / Redirects

Traditional Nuxt applications use the [file system to define routes](https://nuxt.com/docs/getting-started/routing)
based on naming conventions to create dynamic routes.

Drupal already provides a powerful routing system with alias support and redirects that can be managed
by the editors. Drupal also provides multilingual support through translations of aliases.

We wanted to use the power of Drupal's routing system and make it available in Nuxt. In the blökkli starterkit you
can use all the routing features of Drupal in your frontend application. Additionally, you can create routers that only
exists in the frontend.

## Routing

Nuxt provides a feature
called [Cache-all Route](https://nuxt.com/docs/guide/directory-structure/server#catch-all-route). The [Drupal Core
Schema module](https://www.drupal.org/project/graphql_core_schema) provides a GraphQL query called `route` that can be
used to fetch the route information. If no other frontend route matches, we send the path from the incoming requests
to the GraphQL API of Drupal using the `route` query to fetch all the data for the given path.

**GraphQL Schema**

```graphql
extend type Query {
  route(path: String!): Url
}
```

**GraphQL Query**
This is an example of a route query. You don't need to write this query yourself, it is automatically generated
You can use the composable `useDrupalRouteQuery` to fetch the data.

```ts
export default defineNuxtConfig({
  vuepal: {
    drupalRoute: {
      enabled: true,
      // The route queries that will be generated for useDrupalRouteQuery().
      routeQueries: {
        slug: { fragments: ['nodePage'] },
        nodePage: { fragments: ['nodePage'] },
      },
    },
  },
})
```

This will autogenerate a route query like this:

```graphql
query route($path: String!) {
  ...routeQuery

  route(path: $path) {
    ... on EntityUrl {
      entity {
        ...nodePage
      }
    }
  }
}
```

### Advantages

* Leverage Drupal routing
* Support for aliases and redirects based on redirect module
* Multilanguage / multi domain support
* Classic Nuxt routing still possible
* Catch-all route is just as fallback

## Aliases and redirects

As in traditional Drupal sites, the editor can create an alias for a node and multiple redirects per node using the
`redirect` module. The GraphQL query `route` returns the alias and redirect information for a given path.
The Nuxt application will act accordingly and redirect the user to the correct page.

You can use the `[useDrupalRouteQuery()](/frontend/composables#usedrupalroutequery)` composable to get the node information for a
given path.

```vue [pages/[...slug]/index.vue]

<template>
  <NodePage v-if="node" v-bind="node" />
</template>

<script lang="ts" setup>
  defineOptions({
    name: 'PageSlug',
  })

  definePageMeta({
    name: 'drupal-route',
    path: '/:slug(.*)*',
  })

  // Handles redirects and metatags.
  const { entity: node } = await useDrupalRouteQuery('slug')

  await renderPageDependencies()
</script>


```
