# VuePal

VuePal provides a bridge between Drupal and Vue. It comes with a set of components and
composables to make your life easier when working with Drupal.



## Frontend Routing
```ts [nuxt.config.ts]
export default defineNuxtConfig({
  vuepal: {
    frontendRouting: {
      enabled: true,
      langcodes: ['de', 'fr', 'en'],
      outputPath: './../drupal/config/default/frontend_routing.settings.yml',
    },
  },
})
```

With this feature enabled, you can create a static frontend page in Nuxt and still use all the routing features of
Drupal in your frontend application. You can define your aliases in the frontend page using `definePageMeta`. The module
will automatically create a Drupal configuration file that can be imported and processed by the
[Drupal frontend_routing](https://www.drupal.org/project/frontend_routing) module.

```ts [pages/static-page/example.vue]
definePageMeta({
  name: 'static-page-example',
  drupalFrontendRoute: true,
  path: '/static',
  languageMapping: {
    de: 'statisch',
    fr: 'statique',
  },
})
```

## Drupal Route

```ts [nuxt.config.ts]
export default defineNuxtConfig({
  vuepal: {
    drupalRoute: {
      enabled: true,
    },
  },
})
```

This option enables the `useDrupalRoute()` composable.
This composable provides the necessary GraphQL fragment and query to fetch the route data and metatags of a Drupal page.

You can find more information about it on the [composables documentation](/frontend/composables#usedrupalroute).

## Admin Toolbar

```ts [nuxt.config.ts]
export default defineNuxtConfig({
  vuepal: {
    adminToolbar: {
      enabled: true,
    },
  },
})
```

The admin toolbar component fetches the Drupal administration menu and displays it in your frontend application.
![toolbar.png](/content/images/toolbar.png)

### Usage

```vue

<template>
  <ClientOnly>
    <div v-if="drupalUser.accessToolbar && !isEditing">
      <VuepalAdminToolbar :key="language" />
    </div>
  </ClientOnly>
</template>


<script setup lang="ts">
  const route = useRoute()
  const drupalUser = useDrupalUser()
  const language = useCurrentLanguage()
  const isEditing = computed(
    () =>
      !!(route.query.blokkliEditing || route.query.blokkliPreview) &&
      drupalUser.value.accessToolbar,
  )
</script>
```

## LocalTasks

```ts [nuxt.config.ts]
export default defineNuxtConfig({
  vuepal: {
    localTasks: {
      enabled: true,
    },
  },
})
```

The local tasks component fetches the local tasks of a Drupal page and displays them in your frontend application.
![localtasks.png](/content/images/localtasks.png){width=400px}

```vue

<template>
  <ClientOnly>
    <div class="flex">
      <div class="mx-auto w-auto bg-white py-8 xl:min-w-[1174px]">
        <VuepalLocalTasks />
      </div>
    </div>
  </ClientOnly>
</template>
```

## Full Configuration

Example of a full VuePal configuration in the `nuxt.config.ts` file.

```ts
export default defineNuxtConfig({
  vuepal: {
    frontendRouting: {
      enabled: true,
      defaultLanguage: 'de',
      outputPath: './../drupal/config/default/frontend_routing.settings.yml',
    },
    adminToolbar: {
      enabled: true,
    },
    localTasks: {
      enabled: true,
    },
    breadcrumb: {
      enabled: true,
    },
    languageSwitchLinks: {
      enabled: true,
    },
    drupalRoute: {
      enabled: true,
      // The route queries that will be generated for useDrupalRouteQuery().
      routeQueries: {
        slug: { fragments: ['nodePage'] },
        nodeContact: { fragments: ['nodeContact'] },
        nodePage: { fragments: ['nodePage'] },
        nodePressRelease: { fragments: ['nodePressRelease'] },
        nodeCanonical: {
          fragments: ['nodeContact', 'nodePage', 'nodePressRelease'],
        },
      },
    },
    devMode: {
      enabled: true,
      url: `https://${ctx.NUXT_REQUEST_HOST}`,
      forceHttps: true,
    },
  },
})
```
