# Vue Composables

Discover the composables to use in your Vue components and pages.

## `useInitData()`

`useInitData()` is a composable that fetches the initial data for a page. It is only fetched during the first
SSR request and is then stored with useState() passed to the client side hydration.

It usually fetches the main navigation, footer navigation, the translations and the global configuration.
The data is cached using `useDataCache()` and only be fetched once during the first request.

Example usage for loading the main menu:

```ts
const { mainMenuLinks } = useInitData()
```

## `useDrupalRouteQuery()`

`useDrupalRouteQuery()` extracts the route information for a given path from the GraphQL response. It automatically
reacts on aliases or redirects provided by the Drupal. The composable is provided be the `vuepal` module.
This composable provides the necessary GraphQL fragment and query to fetch the route data and metatags of a Drupal page.

The composable is only available, if it is enabled in the vuepal configuration.

```ts
export default defineNuxtConfig({
  vuepal: {
    drupalRoute: {
      enabled: true,
    },
  },
})
```

For more information about the routing and the redirect handling, you can read
the [routing documentation](../frontend/routing).

```vue

<template>
  <NodePage v-if="node" v-bind="node" />
</template>

<script lang="ts" setup>
  defineOptions({
    name: 'PageSlug',
  })

  definePageMeta({
    name: 'drupal-route',
    path: '/:slug(.*)*',
  })

  // Handles redirects and metatags.
  const { entity: node } = await useDrupalRouteQuery('slug')

  await renderPageDependencies()
</script>

```

**GraphQL Query**
The GraphQL query is automatically generated by vuepal module. You just need to fine the required fragments.
It is defined in `.config/vuepal.ts`. More details can be found in
the [VuePal documentation](https://github.com/liip/vuepal?tab=readme-ov-file#drupal-route).

```ts
export default defineNuxtConfigProperty<'vuepal'>((ctx) => {
  return {
    drupalRoute: {
      enabled: true,
      // The route queries that will be generated for useDrupalRouteQuery().
      routeQueries: {
        slug: { fragments: ['nodePage'] },
        // other queries
      },
    },
  }
})
```

## `useBreadcrumb()`

`useBreadcrumb()` provides the breadcrumb links for the current page. It can be called in any component or page,
but has to be wrapped in `<NuxtPageDependency>` if it is displayed outside the `<Nuxt>` component. This is because the
breadcrumb links depend on data that is only available in the `routeQuery`.

The composable is part of VuePal and has to be enabled in the `vuepal` configuration.

```ts
export default defineNuxtConfig({
  vuepal: {
    breadcrumb: {
      enabled: true,
    },
  },
})
```

```vue
    <script setup>
    const breadcrumb = useBreadcrumb()
    </script>
    
    <template>
    <div>
      <NuxtPageDependency>
        <Breadcrumb v-if="showBreadcrumb" :links="breadcrumb" />
      </NuxtPageDependency>
    </div>
    </template>
```

## `useDrupalMessages()`

`useDrupalMessages()` gives access to all fetched Drupal messages globally.

```vue

<template>
  <div v-if="messages.length">
    <div
      v-for="(message, i) in messages"
      :key="i"
    >
      <div v-html="message.message" />
    </div>
  </div>
</template>

<script lang="ts" setup>
  const { messages } = useDrupalMessages()
</script>
```

::source-link
---
source: "composables/useMenu.ts"
---
::

## `useDrupalUser()`

`useDrupalUser()` provides access to the current Drupal user and its roles. It can easily be extended.

```vue

<script setup lang="ts">
  const drupalUser = useDrupalUser()
</script>
```
