/**
 * @file
 * Contains the Drupal attach behavior for the Uppy upload widget.
 */

'use strict';

((Drupal, once) => {

  /**
   * Adds the Uppy upload widget to the media form.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the behavior for the Uppy upload widget.
   */
  Drupal.behaviors.mediaUppyUploadWidget = {
    attach: (context, settings) => {

      const uppySettings = settings.blueBillywigUppySettings || {};

      // Only attach once per form widget.
      once('media-uppy-widget', '.js-form-item-uppy', context).forEach((uploadWidget) => {

        // Get the dashboard container within the widget.
        const dashboardContainer = uploadWidget.querySelector('.js-uppy-dashboard-container');

        // Get the hidden field for upload identifier.
        const uploadIdentifierField = uploadWidget.querySelector('.js-uppy-upload-identifier');

        // Get the hidden field for the media clip ID.
        const mediaClipIdField = uploadWidget.querySelector('.js-uppy-upload-media-clip-id');

        // Get the hidden button to complete the media upload.
        const mediaClipuploadComplete = uploadWidget.querySelector('.js-uppy-upload-media-clip-complete');

        // Keep track of whether we're fetching an upload identifier.
        let fetchingIdentifier = false;

        // Store the OVP data once fetched.
        let uploadData = null;

        // Import the Uppy modules and initialize Uppy.
        const {Uppy, Dashboard, AwsS3} = window.Uppy;
        const uppy = new Uppy({
          autoProceed: false,
          allowMultipleUploadBatches: true,
          restrictions: {
            maxFileSize: uppySettings.maxFileSize || 21474836480,
            allowedFileTypes: uppySettings.allowedFileTypes || ['.mp4', '.mov', '.avi', '.webm', '.ogg', '.mxf', '.mpg', '.mpeg', '.mkv'],
          },
        });

        // Setup the Uppy Dashboard plugin.
        uppy.use(Dashboard, {
          inline: true,
          target: dashboardContainer,
          showProgressDetails: true,
          proudlyDisplayPoweredByUppy: false,
          note: 'Video files only, up to 20GB',
          height: 470,
        });

        // Configure Uppy to upload directly to S3 using pre-signed URLs.
        uppy.use(AwsS3, {
          async getUploadParameters(file) {
            // Wait for the upload identifier.
            while (fetchingIdentifier) {
              await new Promise(resolve => setTimeout(resolve, 100));
            }

            // Fetch the upload identifier from the hidden field.
            const uploadIdentifier = uploadIdentifierField ? uploadIdentifierField.value : '';

            // Ensure we have an upload identifier.
            if (!uploadIdentifier) {
              throw new Error('Upload identifier not available. Please try again.');
            }

            // Request a pre-signed upload URL from the server.
            const response = await fetch(uppySettings.generateUrlEndpoint, {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify({
                filename: file.name,
                filetype: file.type,
                uploadIdentifier: uploadIdentifier,
              }),
            });

            // Handle any errors while fetching the upload URL.
            if (!response.ok) {
              const error = await response.json();
              const errorMessage = error.error || 'Failed to get upload URL';
              const debugInfo = error.received_data ? ` (Received: ${JSON.stringify(error.received_data)})` : '';
              throw new Error(errorMessage + debugInfo);
            }

            // Parse the response to get the upload URL and key.
            const data = await response.json();
            return {
              method: 'PUT',
              url: decodeURIComponent(data.url),
              headers: {
                'Content-Type': file.type,
              },
            };
          },
        });

        // Fetch an upload identifier when a file is added.
        uppy.on('file-added', async (file) => {
          // If we're already fetching or have the identifier, we can skip this.
          if (fetchingIdentifier || uploadData) {
            return;
          }

          // Mark that we're fetching the identifier.
          fetchingIdentifier = true;

          try {
            // Request an upload identifier from the server.
            const response = await fetch(uppySettings.generateUploadIdentifierEndpoint, {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
              },
              body: JSON.stringify({
                filename: file.name,
              }),
            });

            // Handle any errors while fetching the upload identifier.
            const data = await response.json();
            if (data.error) {
              uppy.removeFile(file.id);
              if (uppySettings.debug) {
                console.log(data.error);
              }
            }

            // Store the OVP data and set the upload identifier in the hidden
            // field.
            uploadData = data;
            uploadIdentifierField.value = data.uploadIdentifier;
            mediaClipIdField.value = data.mediaclipId;

            if (uppySettings.debug) {
              console.log('Fetched upload data:');
              console.log(data);
            }
          } catch (error) {

            // When an error occurs, remove the file.
            uppy.removeFile(file.id);
            if (uppySettings.debug) {
              console.log(error.message);
            }

          } finally {

            // We are no longer fetching the identifier.
            fetchingIdentifier = false;

          }
        });

        // If the upload completes, we update the form to show the uploaded
        // clip.
        uppy.on('complete', async (result) => {
          if (result.successful.length > 0 && uploadData && uploadData.mediaclipId) {
            // Trigger an AJAX update using a hidden button.
            mediaClipuploadComplete.dispatchEvent(new MouseEvent('mousedown', { bubbles: true }));
          }
        });


      });

    }
  };

})(Drupal, once);
