<?php

namespace Drupal\blue_billywig;

use BlueBillywig\Request;
use BlueBillywig\Sdk;
use Drupal\blue_billywig\Form\SettingsForm;
use Drupal\blue_billywig\Object\MediaClip;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use GuzzleHttp\RequestOptions;

/**
 * Handles calls to the Blue Billywig API.
 */
class BlueBillywigClient {

  /**
   * The key value store for media clips uploaded to S3 via the Uppy widget.
   *
   * @var string
   */
  public const KEY_VALUE_S3_UPLOADS = 'blue_billywig.s3.uploads';

  /**
   * The time limit after which uncompleted S3 uploads are deleted.
   *
   * @var int
   */
  public const UNCOMPLETED_S3_UPLOADS_LIMIT = 60 * 60 * 4;

  /**
   * The number of search results to return per page.
   *
   * This is the maximum number of results that can be returned in a search
   * request to the Blue Billywig API. The number 32 is chosen since it can be
   * divided evenly by 4, 3 and 2, which allows for an even distribution of
   * items in the media library grid view.
   *
   * @var int
   */
  public const SEARCH_LIMIT = 32;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The key value factory.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueFactoryInterface
   */
  protected KeyValueFactoryInterface $keyValueFactory;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * The Blue Billywig SDK instance.
   *
   * @var \BlueBillywig\Sdk
   */
  protected Sdk $sdk;

  /**
   * Constructs a new BlueBillywigClient instance.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\KeyValueStore\KeyValueFactoryInterface $key_value_factory
   *   The key value factory.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, KeyValueFactoryInterface $key_value_factory, TimeInterface $time) {
    $this->configFactory = $config_factory;
    $this->keyValueFactory = $key_value_factory;
    $this->time = $time;
  }

  /**
   * Gets the Blue Billywig SDK.
   *
   * @return \BlueBillywig\Sdk|null
   *   The Blue Billywig SDK, or null when it is not properly configured.
   */
  protected function getSdk(): ?Sdk {
    // If the SDK is already initialized, return it.
    if (isset($this->sdk)) {
      return $this->sdk;
    }
    $configuration = $this->configFactory->get(SettingsForm::CONFIG_NAME);
    // If the configuration is not set, return null.
    if (!$configuration->get('publication') || !$configuration->get('key') || !$configuration->get('secret')) {
      return NULL;
    }
    return $this->sdk = Sdk::withRPCTokenAuthentication((string) $configuration->get('publication'), (int) $configuration->get('key'), (string) $configuration->get('secret'));
  }

  /**
   * Validates settings for the Blue Billywig SDK.
   *
   * @param string $key
   *   The API key.
   * @param string $secret
   *   The API secret.
   * @param string $publication
   *   The publication ID.
   *
   * @return bool
   *   Whether the API settings are valid or not.
   */
  public function validateApi(string $key, string $secret, string $publication): bool {
    // A key and publication are required to validate the API.
    if (empty($key) || empty($publication)) {
      return FALSE;
    }

    // If the secret is empty, try to validate with the secret stored in
    // configuration.
    if (empty($secret)) {
      $configuration = $this->configFactory->get(SettingsForm::CONFIG_NAME);
      $secret = (string) $configuration->get('secret');
    }

    // Call the playouts endpoint to validate the API settings.
    $sdk = Sdk::withRPCTokenAuthentication($publication, (int) $key, $secret);
    try {
      $response = $sdk->sendRequest(new Request(
        'GET',
        '/sapi/playout'
      ));
      $response->assertIsOk();
    }
    catch (\Exception $e) {
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Loads an item by ID.
   *
   * @param string $id
   *   The ID of the item to load.
   *
   * @return \Drupal\blue_billywig\Object\MediaClip|null
   *   The generated MediaClip object from the API data, or null if the object
   *   was not found.
   */
  public function load(string $id): ?MediaClip {
    // If the SDK is not initialized, return NULL.
    if (!$this->getSdk()) {
      return NULL;
    }

    // Make the request to get the media clip by ID.
    $data = $this->call('/sapi/mediaclip/' . $id);
    return new MediaClip($data, $this->getSdk()->getBaseUri());
  }

  /**
   * Gets the embed code for an item by ID, playout and embed type.
   *
   * @param string $id
   *   The ID of the item to load.
   * @param string $playout
   *   The playout ID to use for the embed code.
   * @param string $embed_type
   *   The embed type to use for the embed code. This needs to be one of the
   *   types defined in SettingsForm::EMBED_TYPES.
   *
   * @return string
   *   The embed code of the item for the provided playout and embed type.
   */
  public function embedCode(string $id, string $playout, string $embed_type): string {
    // If the SDK is not initialized, return an empty string.
    if (!$this->getSdk()) {
      return '';
    }

    // Make the request to get the embed code.
    $data = $this->call('/sapi/embedcode/' . $id . '/' . $playout . '/' . $embed_type);
    return $data['body'] ?? '';
  }

  /**
   * Search for videos by keyword.
   *
   * @param string $keyword
   *   The keyword to search for.
   * @param int $page
   *   An optional page number for the search results.
   * @param int|null $limit
   *   (optional) The limit for the number of results per page. If NULL,
   *   defaults to self::SEARCH_LIMIT.
   *
   * @return array
   *   An array of external media value objects and pager data.
   */
  public function search(string $keyword, int $page = 0, ?int $limit = NULL): array {
    // If the SDK is not initialized, return an empty array.
    if (!$this->getSdk()) {
      return [
        'total' => 0,
        'per_page' => $limit,
        'current_page' => $page,
        'results' => [],
      ];
    }

    $configuration = $this->configFactory->get(SettingsForm::CONFIG_NAME);
    $client_id = (string) $configuration->get('client_id');

    // Make the request to search for media clips.
    $limit = $limit ?? self::SEARCH_LIMIT;
    $request_options = [
      RequestOptions::QUERY => array_filter([
        'q' => $keyword ? 'title:' . $keyword : NULL,
        'fq' => array_filter([
          'status:published',
          $client_id ? 'klantnaam:"' . $client_id . '"' : NULL,
        ]),
        'limit' => $limit,
        'offset' => $limit * $page,
      ]),
    ];
    $data = $this->call('/sapi/mediaclip', $request_options);

    // Add the items to an array of MediaClip objects.
    $items = [];
    foreach ($data['items'] ?? [] as $item) {
      $items[] = new MediaClip($item, $this->getSdk()->getBaseUri());
    }

    // Return the results.
    return [
      'total' => $data['numfound'] ?? 0,
      'per_page' => $limit,
      'current_page' => $page,
      'results' => $items,
    ];
  }

  /**
   * Gets all configured playouts.
   *
   * @return array
   *   An array of configured playouts.
   */
  public function playouts(): array {
    // If the SDK is not initialized, return an empty array.
    if (!$this->getSdk()) {
      return [];
    }

    // Make the request to get the playouts.
    $request_options = [
      RequestOptions::QUERY => [
        'limit' => 100,
        'sort' => 'id asc',
      ],
    ];
    $data = $this->call('/sapi/playout', $request_options);

    // Add the valid playout items to the result.
    $items = [];
    foreach ($data['items'] ?? [] as $item) {
      if (!empty($item['name']) && $item['status'] === 'active') {
        $items[$item['id']] = $item['name'];
      }
    }

    // Return the playouts.
    return $items;
  }

  /**
   * Uploads a new video to the API.
   *
   * @param string $title
   *   The title of the video to upload.
   * @param string $path
   *   The path of the video file to upload.
   *
   * @return \Drupal\blue_billywig\Object\MediaClip|null
   *   The generated MediaClip object from the API data, or null if the file has
   *   not been uploaded.
   */
  public function uploadFile(string $title, string $path): ?MediaClip {
    // If the SDK is not initialized, return NULL.
    if (!$this->getSdk()) {
      return NULL;
    }

    $configuration = $this->configFactory->get(SettingsForm::CONFIG_NAME);
    $client_id = (string) $configuration->get('client_id') ?: NULL;

    try {
      $response = $this->getSdk()->mediaclip->create(array_filter([
        'title' => $title,
        'status' => 'published',
        'klantnaam' => $client_id,
      ]));
      $response->assertIsOk();

      $id = $response->getDecodedBody()['id'];

      $response = $this->getSdk()->mediaclip->initializeUpload($path, $id);
      $response->assertIsOk();

      $data = $response->getDecodedBody();
      $this->getSdk()->mediaclip->helper->executeUpload($path, $data);
    }
    catch (\Exception $e) {
      return NULL;
    }

    // Make the request to get the media clip by ID.
    return $this->load($id);
  }

  /**
   * Registers a new upload and returns the upload identifier and MediaClip ID.
   *
   * @param string $uuid
   *   The UUID for the MediaClip.
   * @param string $original_filename
   *   The original filename of the video.
   *
   * @return array|null
   *   An array containing the upload identifier and MediaClip ID or NULL if
   *   the upload could not be registered.
   *
   * @throws \Exception
   */
  public function registerUpload(string $uuid, string $original_filename = ''): ?array {
    // If the SDK is not initialized, return NULL.
    if (!$this->getSdk()) {
      return NULL;
    }

    $configuration = $this->configFactory->get(SettingsForm::CONFIG_NAME);
    $client_id = (string) $configuration->get('client_id') ?: NULL;

    try {
      $response = $this->getSdk()->mediaclip->create(array_filter([
        'sourceid' => $uuid,
        'title' => $original_filename,
        'status' => 'published',
        'klantnaam' => $client_id,
        'originalfilename' => $original_filename,
      ]));
      $response->assertIsOk();

      $id = $response->getDecodedBody()['id'];

      // If no ID is returned, return NULL.
      if (empty($id)) {
        return NULL;
      }

      $data = $this->call('/sapi/awsupload?autoPublish=false&type=mediaclip&mediaclipId=' . $id);

      $upload_identifier = $data['uploadIdentifier'] ?? NULL;

      // If upload identifier is returned, return NULL.
      if (empty($upload_identifier)) {
        return NULL;
      }
    }
    catch (\Exception $e) {
      return NULL;
    }

    // Mark the media clip as uploading. We do this to either publish the media
    // clip when the media item is saved, or to discard it using cron when the
    // media item creation is cancelled.
    $this->keyValueFactory->get(static::KEY_VALUE_S3_UPLOADS)->set($id, $this->time->getRequestTime());

    // Return the upload identifier and MediaClip ID.
    return [
      'upload_identifier' => $upload_identifier,
      'mediaclip_id' => $id,
    ];
  }

  /**
   * Updates the metadata of a media clip.
   *
   * @param string $id
   *   The ID of the item to update.
   * @param array $data
   *   The data to update for the item.
   *
   * @return bool
   *   TRUE if the media clip was updated successfully, FALSE otherwise.
   */
  public function update(string $id, array $data): bool {
    // If the SDK is not initialized, return FALSE.
    if (!$this->getSdk()) {
      return FALSE;
    }

    try {
      $response = $this->getSdk()->mediaclip->update($id, $data);
      $response->assertIsOk();
    }
    catch (\Exception $e) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Deletes a media clip.
   *
   * @param string $id
   *   The ID of the item to delete.
   *
   * @return bool
   *   TRUE if the media clip was deleted successfully, FALSE otherwise.
   */
  public function delete(string $id): bool {
    // If the SDK is not initialized, return FALSE.
    if (!$this->getSdk()) {
      return FALSE;
    }

    try {
      $this->call('/sapi/mediaclip/' . $id, [], 'DELETE');
    }
    catch (\Exception $e) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Calls the Blue Billywig API.
   *
   * @param string $endpoint
   *   The API endpoint to call.
   * @param array $request_options
   *   The request options to use for the API call.
   * @param string $method
   *   The HTTP method to use for the API call.
   * @param \BlueBillywig\Sdk|null $sdk
   *   The SDK instance to use for the API call. If NULL, the default SDK will
   *   be used.
   *
   * @return array
   *   The response data from the API.
   */
  protected function call(string $endpoint, array $request_options = [], string $method = 'GET', ?Sdk $sdk = NULL): array {
    try {
      $response = $this->getSdk()->sendRequest(new Request(
        $method,
        $endpoint
      ), $request_options);
      $response->assertIsOk();

      return $response->getDecodedBody();
    }
    catch (\Exception $e) {
      return [];
    }
  }

}
