<?php

namespace Drupal\blue_billywig\Controller;

use Aws\S3\S3Client;
use Drupal\blue_billywig\BlueBillywigClient;
use Drupal\blue_billywig\Form\SettingsForm;
use Drupal\Component\Serialization\Json;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for S3 upload operations.
 */
class S3UploadController extends ControllerBase {

  use LoggerChannelTrait;

  /**
   * Allowed video MIME types for upload.
   *
   * @var string[]
   */
  const ALLOWED_VIDEO_EXTENSIONS = [
    'mp4',
    'mov',
    'avi',
    'webm',
    'ogv',
    'mxf',
    'mpeg',
    'mkv',
  ];

  /**
   * Allowed video MIME types for upload.
   *
   * @var string[]
   */
  const ALLOWED_VIDEO_TYPES = [
    'video/mp4',
    'video/quicktime',
    'video/x-msvideo',
    'video/webm',
    'video/ogg',
    'application/mxf',
    'video/x-mxf',
    'video/mpeg',
    'video/x-matroska',
    'video/mkv',
  ];

  /**
   * The Blue Billywig client.
   *
   * @var \Drupal\blue_billywig\BlueBillywigClient
   */
  protected BlueBillywigClient $client;

  /**
   * The UUID service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected UuidInterface $uuid;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $instance = parent::create($container);
    $instance->client = $container->get('blue_billywig.client');
    $instance->uuid = $container->get('uuid');
    return $instance;
  }

  /**
   * Generates a presigned URL for S3 upload.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The incoming request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the presigned URL or error message.
   */
  public function generatePresignedUrl(Request $request): JsonResponse {
    $configuration = $this->config(SettingsForm::CONFIG_NAME);
    // If the required configuration is not set, return an error.
    if (!$configuration->get('aws_access_key_id') || !$configuration->get('aws_secret_access_key') || !$configuration->get('aws_s3_bucket_url')) {
      return new JsonResponse([
        'error' => 'The AWS access key ID, secret or bucket URL for uploads are not configured correctly.',
      ], 400);
    }

    $data = Json::decode($request->getContent());
    $filename = $data['filename'] ?? '';
    $filetype = $data['filetype'] ?? '';
    $upload_identifier = $data['uploadIdentifier'] ?? '';

    // Validate the upload identifier.
    if (empty($upload_identifier)) {
      return new JsonResponse([
        'error' => 'The upload identifier is required.',
        'received_data' => $data,
      ], 400);
    }

    // Validate file type (only allow video files).
    if (!in_array($filetype, static::ALLOWED_VIDEO_TYPES, TRUE)) {
      return new JsonResponse([
        'error' => 'Invalid file type. Only video files are allowed.',
      ], 400);
    }

    // Sanitize the filename.
    $filename = preg_replace('/[^a-zA-Z0-9._-]/', '', $filename);
    $prefix = trim($configuration->get('aws_s3_upload_prefix') ?? 'upload/' . $configuration->get('publication'), '/') . '/';
    $key = $prefix . uniqid('', TRUE) . '_' . $filename;

    try {
      // Initialize S3 client settings.
      $s3_config = array_filter([
        'version' => 'latest',
        'region' => $configuration->get('aws_s3_region') ?? 'eu-west-1',
        'credentials' => [
          'key' => $configuration->get('aws_access_key_id'),
          'secret' => $configuration->get('aws_secret_access_key'),
        ],
      ]);

      // Add endpoint if configured (for MinIO or other S3-compatible services).
      $endpoint = $configuration->get('aws_s3_endpoint') ?? '';
      if (!empty($endpoint)) {
        $s3_config['endpoint'] = $endpoint;
      }

      // Create the S3 client.
      $s3_client = new S3Client($s3_config);

      // Create presigned request command.
      $request_cmd = $s3_client->getCommand('PutObject', [
        'Bucket' => str_replace('s3://', '', $configuration->get('aws_s3_bucket_url')),
        'Key' => $key,
        'ContentType' => $filetype,
        'ACL' => 'private',
        'Metadata' => [
          'uploadidentifier' => $upload_identifier,
        ],
      ]);

      // Generate presigned URL (valid for 15 minutes)
      $request = $s3_client->createPresignedRequest($request_cmd, '+15 minutes');
      return new JsonResponse([
        'url' => (string) $request->getUri(),
        'key' => $key,
        'bucket' => str_replace('s3://', '', $configuration->get('aws_s3_bucket_url')),
      ]);
    }
    catch (\Exception $e) {
      $this->getLogger('blue_billywig')->error('Error generating presigned URL: @error', [
        '@error' => $e->getMessage(),
      ]);
      return new JsonResponse([
        'error' => 'Failed to generate presigned URL.',
      ], 500);
    }
  }

  /**
   * Generates an upload identifier from Blue Billywig.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The incoming request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the upload identifier or error message.
   */
  public function generateUploadIdentifier(Request $request): JsonResponse {
    try {
      $data = Json::decode($request->getContent());
      $filename = $data['filename'] ?? '';

      // Generate a UUID for the source ID.
      $uuid = $this->uuid->generate();

      // Register the file upload in Blue Billywig.
      $result = $this->client->registerUpload($uuid, $filename);

      // If the result does not contain an upload identifier or mediaclip ID,
      // return an error.
      if (empty($result['upload_identifier']) || empty($result['mediaclip_id'])) {
        throw new \Exception('Invalid response from Blue Billywig API.');
      }

      // Return the upload identifier and mediaclip ID.
      return new JsonResponse([
        'uploadIdentifier' => $result['upload_identifier'],
        'mediaclipId' => $result['mediaclip_id'],
        'uuid' => $uuid,
      ]);
    }
    catch (\Exception $e) {
      $this->getLogger('blue_billywig')->error('Error generating upload identifier: @error', [
        '@error' => $e->getMessage(),
      ]);
      return new JsonResponse([
        'error' => 'Failed to generate upload identifier.',
      ], 500);
    }
  }

}
