<?php

namespace Drupal\blue_billywig\Form;

use Drupal\blue_billywig\BlueBillywigClient;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a form for configuring the Blue Billywig module.
 */
class SettingsForm extends ConfigFormBase {

  use StringTranslationTrait;

  /**
   * The configuration name for Blue Billywig settings.
   *
   * @var string
   */
  public const CONFIG_NAME = 'blue_billywig.settings';

  /**
   * The list of Blue Billywig embed types.
   *
   * @var array
   */
  public const EMBED_TYPES = [
    'javascript' => 'JavaScript',
    'iframe' => 'Iframe',
  ];

  /**
   * The Blue Billywig client.
   *
   * @var \Drupal\blue_billywig\BlueBillywigClient
   */
  protected BlueBillywigClient $client;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->client = $container->get('blue_billywig.client');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getEditableConfigNames(): array {
    return ['blue_billywig.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'blue_billywig_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    $config = $this->config(self::CONFIG_NAME);

    $form['api'] = [
      '#type' => 'details',
      '#title' => $this->t('API Configuration'),
      '#description' => $this->t('The API key and secret can be created in your Blue Billywig account. Navigate to "Publication settings" > "Secrets" to create a new API key.'),
      '#open' => TRUE,
    ];
    $form['api']['publication'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Publication (sub)domain'),
      '#description' => $this->t('Enter the subdomain for the publication. For example, when your publication domain is "https://example.bbvms.com", enter "example".'),
      '#default_value' => $config->get('publication') ?: '',
      '#required' => TRUE,
    ];
    $form['api']['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client identifier (optional)'),
      '#description' => $this->t('Enter the client identifier for the publication. This filters the API results for the publication.'),
      '#default_value' => $config->get('client_id') ?: '',
    ];
    $form['api']['key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Enter your API key. This is the ID related to the API secret.'),
      '#default_value' => $config->get('key') ?: '',
      '#required' => TRUE,
    ];
    $form['api']['secret'] = [
      '#type' => 'password',
      '#title' => $this->t('API Secret'),
      '#description' => $this->t('Enter your API secret. Leave blank to keep the existing value.'),
      '#default_value' => '',
      '#attributes' => [
        'placeholder' => $config->get('secret') ? $this->t('Secret is set (hidden)') : $this->t('No secret set'),
        'autocomplete' => 'off',
      ],
    ];

    $form['aws'] = [
      '#type' => 'details',
      '#title' => $this->t('AWS Upload settings'),
      '#description' => $this->t('To upload large video files, it is recommended to use the Uppy upload widget. This will send the uploads directly to Blue Billywig, avoiding any limits that might exist on the server.<br>To set this up, create an AWS access key ID and secret in your Blue Billywig account.<br>Navigate to "Publication settings" > "Upload credentials" to create a new key.<br>Once the credentials are entered, the Uppy upload widget will automatically be available to add new videos.'),
      '#open' => (bool) $config->get('aws_access_key_id'),
    ];
    $form['aws']['aws_access_key_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWS access key ID'),
      '#description' => $this->t('Enter your AWS access key ID.'),
      '#default_value' => $config->get('aws_access_key_id') ?: '',
    ];
    $form['aws']['aws_secret_access_key'] = [
      '#type' => 'password',
      '#title' => $this->t('AWS secret key'),
      '#description' => $this->t('Enter your AWS secret access key. Leave blank to keep the existing value.'),
      '#default_value' => '',
      '#attributes' => [
        'placeholder' => $config->get('secret') ? $this->t('Secret is set (hidden)') : $this->t('No secret set'),
        'autocomplete' => 'off',
      ],
    ];
    $form['aws']['aws_s3_bucket_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWS S3 bucket url'),
      '#description' => $this->t('Enter your AWS S3 bucket url.'),
      '#default_value' => $config->get('aws_s3_bucket_url') ?: '',
    ];
    $form['aws']['aws_s3_upload_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWS S3 upload prefix'),
      '#description' => $this->t('Enter the AWS S3 upload prefix. When left empty, this will default to "/upload/[publication]".'),
      '#default_value' => $config->get('aws_s3_upload_prefix') ?: '',
      '#attributes' => [
        'placeholder' => $config->get('publication') ? '/upload/' . $config->get('publication') : '/upload/[publication]/',
      ],
    ];
    $form['aws']['aws_s3_region'] = [
      '#type' => 'textfield',
      '#title' => $this->t('AWS S3 region'),
      '#description' => $this->t('Enter the AWS S3 region for the bucket. Defaults to "eu-west-1" when left empty.'),
      '#default_value' => $config->get('aws_s3_region') ?: 'eu-west-1',
    ];
    // Allow debug information to be logged for AWS uploads.
    $form['aws']['aws_debug'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable debug logging for AWS S3 uploads to the browser console and Drupal log'),
      '#default_value' => $config->get('aws_debug') ?: FALSE,
    ];

    $form['options'] = [
      '#type' => 'details',
      '#title' => $this->t('Display options'),
      '#open' => TRUE,
    ];
    $form['options']['playout'] = [
      '#type' => 'select',
      '#title' => $this->t('Playout'),
      '#default_value' => $config->get('playout') ?: '',
      '#options' => $this->client->playouts(),
    ];
    $form['options']['embed_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Embed type'),
      '#default_value' => $config->get('embed_type') ?: '',
      '#options' => self::EMBED_TYPES,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // Validate the API key, secret and publication.
    if (!$this->client->validateApi($form_state->getValue('key'), $form_state->getValue('secret'), $form_state->getValue('publication'))) {
      $form_state->setError($form['api'], $this->t('The API key, secret and/or publication are not valid. Please check the settings.'));
      $form_state->setErrorByName('key');
      $form_state->setErrorByName('secret');
      $form_state->setErrorByName('publication');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $settings = $this->config('blue_billywig.settings')
      ->set('publication', str_replace(['https://', 'http://', '.bbvms.com'], '', $form_state->getValue('publication')))
      ->set('key', $form_state->getValue('key'))
      ->set('client_id', $form_state->getValue('client_id'))
      ->set('aws_access_key_id', $form_state->getValue('aws_access_key_id'))
      ->set('aws_s3_bucket_url', $form_state->getValue('aws_s3_bucket_url'))
      ->set('aws_s3_upload_prefix', trim($form_state->getValue('aws_s3_upload_prefix'), '/'))
      ->set('aws_debug', $form_state->getValue('aws_debug'))
      ->set('playout', $form_state->getValue('playout'))
      ->set('embed_type', $form_state->getValue('embed_type'));

    // Only update the secrets when they are provided.
    if ($form_state->getValue('secret')) {
      $settings->set('secret', $form_state->getValue('secret'));
    }
    if ($form_state->getValue('aws_secret_access_key')) {
      $settings->set('aws_secret_access_key', $form_state->getValue('aws_secret_access_key'));
    }

    // Save the settings.
    $settings->save();

    // Add a message.
    $this->messenger()->addStatus($this->t('The configuration has been saved.'));
  }

}
