<?php

namespace Drupal\blue_billywig;

use Drupal\blue_billywig\Object\MediaClip;
use Drupal\Core\Ajax\AjaxHelperTrait;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InsertCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\media\MediaSourceManager;

/**
 * Provides search results for the media library add form.
 */
class MediaLibrarySearch {

  use AjaxHelperTrait;
  use StringTranslationTrait;

  /**
   * The Blue Billywig client.
   *
   * @var \Drupal\blue_billywig\BlueBillywigClient
   */
  protected BlueBillywigClient $client;

  /**
   * The pager manager.
   *
   * @var \Drupal\Core\Pager\PagerManagerInterface
   */
  protected PagerManagerInterface $pagerManager;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The media source manager.
   *
   * @var \Drupal\media\MediaSourceManager
   */
  protected MediaSourceManager $mediaSourceManager;

  /**
   * Constructs a new MediaLibrarySearch instance.
   *
   * @param \Drupal\blue_billywig\BlueBillywigClient $client
   *   The Blue Billywig client.
   * @param \Drupal\Core\Pager\PagerManagerInterface $pager_manager
   *   The pager manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\media\MediaSourceManager $media_source_manager
   *   The media source manager.
   */
  public function __construct(BlueBillywigClient $client, PagerManagerInterface $pager_manager, ConfigFactoryInterface $config_factory, MediaSourceManager $media_source_manager) {
    $this->client = $client;
    $this->pagerManager = $pager_manager;
    $this->configFactory = $config_factory;
    $this->mediaSourceManager = $media_source_manager;
  }

  /**
   * Gets the search results for the media library.
   *
   * @param string $keyword
   *   The search keyword.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse|array
   *   An AJAX response containing the search results.
   */
  public function searchResults(string $keyword): AjaxResponse|array {
    $results = $this->buildSearchResults($keyword);
    if ($this->isAjax()) {
      $response = new AjaxResponse();
      $response->addCommand(new InsertCommand('#media-library-results', $results));
      return $response;
    }
    return $results;
  }

  /**
   * Gets search results for the media library.
   *
   * @param string $keyword
   *   The search keyword.
   *
   * @return array
   *   An array of video results.
   */
  public function buildSearchResults(string $keyword): array {
    $build = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'media-library-results',
        'class' => [
          'media-library-add-form__result-wrapper',
        ],
      ],
    ];

    $build['results'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => [
          'media-library-view',
          'media-library-views-form__rows',
        ],
      ],
      '#tree' => TRUE,
    ];

    // Fetch the results for the current page.
    $results = $this->client->search($keyword, $this->pagerManager->findPage());

    // Update the pager.
    $this->pagerManager->createPager($results['total'], $results['per_page']);

    foreach ($results['results'] as $mediaclip) {
      // Check if the result item is valid.
      if (!$mediaclip instanceof MediaClip || !$mediaclip->getId()) {
        continue;
      }

      // Render the media clip in the results.
      $id = $mediaclip->getId();
      $title = $mediaclip->getTitle() ?: $mediaclip->getDescription() ?: $id;
      $build['results'][$id] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => [
            'media-library-item',
            'media-library-item--grid',
            'js-media-library-import-item',
            'js-click-to-select',
          ],
        ],
      ];

      // Add a select field to allow selecting the media item.
      $build['results'][$id]['select_wrapper'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => [
            'media-library-item__click-to-select-checkbox',
            'js-click-to-select-checkbox',
          ],
        ],
      ];
      $build['results'][$id]['select_wrapper']['select'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Select @label', [
          '@label' => $title,
        ]),
        '#title_display' => 'invisible',
        '#return_value' => $id,
      ];

      // Create a wrapper for the media item that can be clicked to select it.
      $build['results'][$id]['item_wrapper'] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => [
            'media-library-item__click-to-select-trigger',
            'js-click-to-select-trigger',
          ],
        ],
      ];

      // Add the preview and title to the item wrapper.
      $thumbnail_url = $mediaclip->getThumbnailUrl() ?: $this->configFactory->get('media.settings')->get('icon_base_uri') . '/' . $this->mediaSourceManager->getDefinition('blue_billywig')['default_thumbnail_filename'];
      $build['results'][$id]['item_wrapper']['preview_wrapper'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['media-library-item__preview-wrapper']],
        'preview' => [
          '#type' => 'container',
          '#attributes' => ['class' => ['media-library-item__preview']],
          'thumbnail' => [
            '#type' => 'container',
            '#attributes' => ['class' => ['field--name-thumbnail']],
            'image' => [
              '#theme' => 'image',
              '#uri' => $thumbnail_url,
              '#alt' => $mediaclip->getThumbnailUrl() ? $this->t('The thumbnail for @title', [
                '@title' => $title,
              ]) : $this->t('The default video thumbnail icon.'),
            ],
          ],
        ],
      ];
      $build['results'][$id]['item_wrapper']['preview_wrapper']['title_wrapper'] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['media-library-item__attributes']],
        'title' => [
          '#type' => 'html_tag',
          '#tag' => 'div',
          '#value' => $title,
          '#attributes' => ['class' => ['media-library-item__name']],
        ],
      ];
    }

    $build['pager'] = [
      '#type' => 'pager',
      '#route_name' => 'blue_billywig.media_library_search',
      '#route_parameters' => [
        'keyword' => $keyword,
      ],
    ];

    return $build;
  }

}
