<?php

namespace Drupal\blue_billywig\Object;

/**
 * Value object for videos from the Blue Billywig API.
 */
class MediaClip {

  /**
   * The default width of the thumbnail image.
   *
   * @var int
   */
  public const THUMBNAIL_WIDTH = 400;

  /**
   * The default height of the thumbnail image.
   *
   * @var int
   */
  public const THUMBNAIL_HEIGHT = 225;

  /**
   * The base URL of the Blue Billywig assets.
   *
   * @var string
   */
  protected string $baseUrl;

  /**
   * The unique ID of the media clip.
   *
   * @var string
   */
  protected string $id;

  /**
   * The title of the media clip.
   *
   * @var string
   */
  protected string $title;

  /**
   * The description of the media clip.
   *
   * @var string
   */
  protected string $description;

  /**
   * The thumbnail of the media clip.
   *
   * @var array|null
   */
  protected ?array $thumbnail;

  /**
   * The original filename of the media clip.
   *
   * @var string|null
   */
  protected ?string $filename;

  /**
   * Constructs a MediaClip object.
   *
   * @param array $data
   *   The data provided by the Blue Billywig API.
   * @param string $base_url
   *   The base URL for the Blue Billywig API. We use this to relative URLs.
   */
  public function __construct(array $data, string $base_url) {
    $this->baseUrl = trim($base_url, '/');
    $this->id = $data['id'] ?? NULL;
    $this->title = $data['title'] ?? $data['originalfilename'] ?? '';
    $this->description = $data['description'] ?? '';
    $this->thumbnail = !empty($data['thumbnails'][0]['src']) ? [
      'src' => $data['thumbnails'][0]['src'],
      'width' => $data['thumbnails'][0]['width'] ?? NULL,
      'height' => $data['thumbnails'][0]['height'] ?? NULL,
    ] : NULL;
    $this->filename = $data['originalfilename'] ?? NULL;
  }

  /**
   * Gets the unique ID of the media clip.
   *
   * @return string
   *   The unique ID of the media clip.
   */
  public function getId(): string {
    return $this->id;
  }

  /**
   * Gets the title of the media clip.
   *
   * @return string
   *   The title of the media clip.
   */
  public function getTitle(): string {
    return $this->title;
  }

  /**
   * Gets the description of the media clip.
   *
   * @return string
   *   The description of the media clip.
   */
  public function getDescription(): string {
    return $this->description;
  }

  /**
   * Gets the thumbnail URL of the media clip.
   *
   * @param int|null $width
   *   The requested width of the thumbnail image.
   *
   * @return string|null
   *   The URL of the thumbnail image, or NULL if no thumbnail is available.
   */
  public function getThumbnailUrl(?int $width = NULL): ?string {
    // Check if the thumbnail data is available.
    if (empty($this->thumbnail) || empty($this->thumbnail['src'])) {
      return NULL;
    }
    // If no width is provided, use the default thumbnail width to create an
    // optimized thumbnail image.
    $width = $width ?? self::THUMBNAIL_WIDTH;
    $height = self::THUMBNAIL_HEIGHT;
    if (!empty($this->thumbnail['width']) && !empty($this->thumbnail['height'])) {
      $height = !empty($width) ? ($width * $this->thumbnail['height']) / $this->thumbnail['width'] : $this->thumbnail['height'];
    }
    return $this->baseUrl . '/image/' . $width . '/' . $height . '/' . $this->thumbnail['src'];
  }

  /**
   * Gets the original filename of the media clip.
   *
   * @return string|null
   *   The original filename of the media clip, or NULL if not available.
   */
  public function getFilename(): ?string {
    return $this->filename;
  }

}
