<?php

namespace Drupal\blue_billywig\Plugin\Field\FieldFormatter;

use Drupal\blue_billywig\BlueBillywigClient;
use Drupal\blue_billywig\Form\SettingsForm;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\StringFormatter;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'blue_billywig_embed_code' formatter.
 */
#[FieldFormatter(
  id: 'blue_billywig_embed_code',
  label: new TranslatableMarkup('Blue Billywig embed code'),
  field_types: [
    'blue_billywig_id',
  ],
)]
class BlueBillywigFormatter extends StringFormatter {

  /**
   * The Blue Billywig client.
   *
   * @var \Drupal\blue_billywig\BlueBillywigClient
   */
  protected BlueBillywigClient $client;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->client = $container->get('blue_billywig.client');
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'playout' => NULL,
      'embed_type' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $form['playout'] = [
      '#type' => 'select',
      '#title' => $this->t('Playout'),
      '#default_value' => $this->getSetting('playout'),
      '#options' => ['' => $this->t('- Site defaults -')] + $this->client->playouts(),
    ];
    $form['embed_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Embed type'),
      '#default_value' => $this->getSetting('embed_type'),
      '#options' => ['' => $this->t('- Site defaults -')] + SettingsForm::EMBED_TYPES,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = parent::settingsSummary();
    if ($this->getSetting('playout')) {
      $playouts = $this->client->playouts();
      $summary[] = $this->t('Playout: @playout', ['@playout' => $playouts[$this->getSetting('playout')] ?? $this->t('- Site defaults -')]);
    }
    if ($this->getSetting('embed_type')) {
      $summary[] = $this->t('Embed type: @embed_type', ['@embed_type' => SettingsForm::EMBED_TYPES[$this->getSetting('embed_type')] ?? $this->t('- Site defaults -')]);
    }
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  protected function viewValue(FieldItemInterface $item) {
    // Fetch the embed code from the Blue Billywig client.
    $playout_id = $this->getSetting('playout') ?: $this->configFactory->get(SettingsForm::CONFIG_NAME)->get('playout');
    $embed_type = $this->getSetting('embed_type') ?: $this->configFactory->get(SettingsForm::CONFIG_NAME)->get('embed_type');
    return [
      '#markup' => Markup::create($this->client->embedCode($item->value, $this->client->playouts()[$playout_id] ?? 'default', $embed_type)),
    ];
  }

}
