<?php

namespace Drupal\blue_billywig\Plugin\Field\FieldWidget;

use Drupal\blue_billywig\BlueBillywigClient;
use Drupal\blue_billywig\Controller\S3UploadController;
use Drupal\blue_billywig\Form\SettingsForm;
use Drupal\blue_billywig\Object\MediaClip;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\StringTextfieldWidget;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\media\MediaInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'blue_billywig_id' widget.
 */
#[FieldWidget(
  id: 'blue_billywig_id',
  label: new TranslatableMarkup('Blue Billywig ID'),
  field_types: ['blue_billywig_id'],
)]
class BlueBillywigWidget extends StringTextfieldWidget {

  /**
   * The Blue Billywig client.
   *
   * @var \Drupal\blue_billywig\BlueBillywigClient
   */
  protected BlueBillywigClient $client;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->client = $container->get('blue_billywig.client');
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // For existing media items, render a preview of the media item.
    $media = $items->getEntity();
    $source_field_value = $media instanceof MediaInterface ? $media->getSource()->getSourceFieldValue($media) : NULL;
    if (!empty($source_field_value) && !$media->isNew()) {
      $element['preview'] = $this->renderVideoPreview($source_field_value);
      return $element;
    }

    // Fetch the module configuration.
    $configuration = $this->configFactory->get(SettingsForm::CONFIG_NAME);

    // Create an ID from the parents to make sure each widget is unique.
    $field_name = $this->fieldDefinition->getName();
    $parents = $form['#parents'];
    $id_suffix = $parents ? '-' . implode('-', $parents) : '';
    $wrapper_id = $field_name . '-blue-billywig-wrapper' . $id_suffix;

    // For new media items, we show the Uppy Upload Widget when the AWS settings
    // are configured. If not, we fall back to a regular upload field.
    if (!empty($configuration->get('aws_access_key_id')) && !empty($configuration->get('aws_secret_access_key')) && !empty($configuration->get('aws_s3_bucket_url'))) {
      $element['uppy_upload'] = $this->buildUppyUploadWidget($wrapper_id);
      // Hide the original input field.
      $element['value']['#type'] = 'hidden';
    }
    else {
      $element['file_upload'] = $this->buildFileUploadWidget($element['value']['#description'] ?? NULL);
      // Add a validator to verify if the file can be uploaded.
      $element['value']['#element_validate'][] = [static::class, 'uploadFile'];
      // Hide the original input field.
      $element['value']['#access'] = FALSE;
    }

    // Set the wrapper ID for AJAX updates.
    $element['#prefix'] = '<div id="' . $wrapper_id . '">';
    $element['#suffix'] = '</div>';

    // Add the wrapper for the widget.
    return $element;
  }

  /**
   * Form validation handler for Blue Billywig ID elements.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public static function uploadFile(array $element, FormStateInterface $form_state): void {
    // Verify we have a valid field name.
    if (empty($element['#parents'][0])) {
      $form_state->setError($element, t('The file upload field is not properly configured.'));
      return;
    }

    // Fetch the path to the uploaded file.
    $uploaded_file = \Drupal::request()->files->get('files', [])[$element['#parents'][0]] ?? [];
    if (empty($uploaded_file)) {
      $form_state->setError($element, t('No file was uploaded.'));
      return;
    }

    // Temprarily move the uploaded file to a temp location using the filename.
    // This is necessary because the Blue Billywig SDK uses the filename from
    // the upload path and uses it to detect the file type.
    $tmp_path = 'temporary://' . $uploaded_file->getClientOriginalName();
    if (!\Drupal::service('file_system')->moveUploadedFile($uploaded_file->getRealPath(), $tmp_path)) {
      $form_state->setError($element, t('The file could not be uploaded. Please try again.'));
      return;
    }

    // Try to upload the file to the API and get the created Blue Billywig ID.
    $media_name = !empty($form_state->getValue('name')[0]['value']) ? $form_state->getValue('name')[0]['value'] : $uploaded_file->getClientOriginalName();
    $mediaclip = \Drupal::service('blue_billywig.client')->uploadFile($media_name, \Drupal::service('file_system')->realpath($tmp_path));

    // Remove the temporary file after upload.
    \Drupal::service('file_system')->delete($tmp_path);

    if (!$mediaclip instanceof MediaClip) {
      $form_state->setError($element, t('The file could not be uploaded. Please try again.'));
      return;
    }

    // If the ID is valid, set it back to the element.
    $form_state->setValueForElement($element, $mediaclip->getId());
  }

  /**
   * Renders a video preview for the given Blue Billywig ID.
   *
   * @param string $blue_billywig_id
   *   The Blue Billywig media clip ID.
   *
   * @return array
   *   A renderable array containing the video preview.
   */
  protected function renderVideoPreview(string $blue_billywig_id): array {
    $configuration = $this->configFactory->get(SettingsForm::CONFIG_NAME);
    $playout_id = $configuration->get('playout');
    $embed_type = $configuration->get('embed_type');
    return [
      '#markup' => Markup::create($this->client->embedCode($blue_billywig_id, $this->client->playouts()[$playout_id] ?? 'default', $embed_type)),
    ];
  }

  /**
   * Builds a file upload widget form element.
   *
   * @param string $description
   *   The description for the file upload widget.
   *
   * @return array
   *   A renderable array containing the file upload widget.
   */
  protected function buildFileUploadWidget(string $description): array {
    return [
      '#type' => 'file',
      '#title' => $this->t('Upload a new video'),
      '#description' => $description,
      '#upload_validators' => [
        'FileExtension' => [
          'extensions' => implode(' ', S3UploadController::ALLOWED_VIDEO_EXTENSIONS),
        ],
      ],
    ];
  }

  /**
   * Builds the Uppy upload widget form element.
   *
   * @param string $wrapper_id
   *   The ID of the wrapper element for AJAX updates.
   *
   * @return array
   *   A renderable array containing the Uppy upload widget.
   */
  protected function buildUppyUploadWidget(string $wrapper_id): array {
    $configuration = $this->configFactory->get(SettingsForm::CONFIG_NAME);
    // Create the container for the Uppy upload widget.
    return [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['form-item', 'form-item--uppy', 'js-form-item-uppy'],
      ],
      // Hidden field to store the Blue Billywig media clip ID.
      'mediaclip_id' => [
        '#type' => 'hidden',
        '#attributes' => [
          'class' => 'js-uppy-upload-media-clip-id',
        ],
      ],
      // Hidden field to store the S3 upload identifier.
      'upload_identifier' => [
        '#type' => 'hidden',
        '#attributes' => [
          'class' => 'js-uppy-upload-identifier',
        ],
      ],
      // Hidden button to trigger the upload process.
      'uppy_completed' => [
        '#type' => 'button',
        '#value' => $this->t('Complete file upload'),
        '#attributes' => [
          'class' => ['js-uppy-upload-media-clip-complete', 'js-hide'],
        ],
        '#ajax' => [
          'callback' => [$this, 'updateUppyWidget'],
          'wrapper' => $wrapper_id,
          'progress' => [
            'type' => 'throbber',
            'message' => $this->t('Adding selection.'),
          ],
        ],
        '#limit_validation_errors' => [],
      ],
      // Create a container for the Uppy dashboard.
      'dashboard' => [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['uppy-dashboard-container', 'js-uppy-dashboard-container'],
        ],
      ],
      '#attached' => [
        'library' => [
          'blue_billywig/uppy',
        ],
        'drupalSettings' => [
          'blueBillywigUppySettings' => [
            'generateUrlEndpoint' => Url::fromRoute('blue_billywig.s3.generate_presigned_url')->toString(),
            'generateUploadIdentifierEndpoint' => Url::fromRoute('blue_billywig.s3.generate_upload_identifier')->toString(),
            // Max file size is 20MB.
            'maxFileSize' => 1024 * 1024 * 1024 * 20,
            'allowedFileTypes' => array_map(static function ($ext) {
              return '.' . $ext;
            }, S3UploadController::ALLOWED_VIDEO_EXTENSIONS),
            'debug' => $configuration->get('aws_debug') ?? FALSE,
          ],
        ],
      ],
    ];
  }

  /**
   * AJAX callback to update the widget when a file is uploaded using Uppy.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The updated form element.
   */
  public function updateUppyWidget(array $form, FormStateInterface $form_state): array {
    $triggering_element = $form_state->getTriggeringElement();

    // Set the nesting level for the triggering element.
    $length = -2;

    // Get the form element that triggered the AJAX callback.
    if (count($triggering_element['#array_parents']) < abs($length)) {
      throw new \LogicException('The element that triggered the widget update was at an unexpected depth. Triggering element parents were: ' . implode(',', $triggering_element['#array_parents']));
    }

    // Retrieve the form element using its parents.
    $parents = array_slice($triggering_element['#array_parents'], 0, $length);
    $element = NestedArray::getValue($form, $parents);

    // We need to use the actual user input, since when #limit_validation_errors
    // is used, the non validated user input is not added to the form state.
    // @see FormValidator::handleErrorsWithLimitedValidation()
    $path = $element['#parents'];
    $user_input = $form_state->getUserInput();
    $values = NestedArray::getValue($user_input, $path);

    // Set the value of the form element to the selected media clip ID.
    if (!empty($values['uppy_upload']['mediaclip_id'])) {
      $element['value']['#value'] = $values['uppy_upload']['mediaclip_id'];
      // Remove the Uppy Upload Widget and show a preview.
      unset($element['uppy_upload']);
      $element['preview'] = $this->renderVideoPreview($values['uppy_upload']['mediaclip_id']);
    }

    $form_state->setRebuild();
    return $element;
  }

}
