# Drupal 12 Compatibility Fix

## Issue
GitLab CI/CD pipeline was failing with deprecation warnings that will become errors in Drupal 12:

```
Functional/FunctionalJavascript test classes must specify the #[RunTestsInSeparateProcesses] attribute,
not doing so is deprecated in drupal:11.3.0 and is throwing an exception in drupal:12.0.0.
See https://www.drupal.org/node/3548485
```

```
Kernel test classes must specify the #[RunTestsInSeparateProcesses] attribute,
not doing so is deprecated in drupal:11.3.0 and will throw an exception in drupal:12.0.0.
See https://www.drupal.org/node/3548485
```

## Root Cause
Drupal 11.3+ requires all Functional and Kernel test classes to have the `#[RunTestsInSeparateProcesses]`
attribute to ensure tests run in isolated processes. This prevents test pollution and improves reliability.

## Solution
Added the `#[RunTestsInSeparateProcesses]` attribute to both test classes:

### 1. Functional Test (`tests/src/Functional/BodyClassTest.php`)

**Added**:
```php
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

#[RunTestsInSeparateProcesses]
class BodyClassTest extends BrowserTestBase {
```

### 2. Kernel Test (`tests/src/Kernel/BodyClassKernelTest.php`)

**Added**:
```php
use PHPUnit\Framework\Attributes\RunTestsInSeparateProcesses;

#[RunTestsInSeparateProcesses]
class BodyClassKernelTest extends KernelTestBase {
```

## Impact
- ✅ Tests now comply with Drupal 11.3+ requirements
- ✅ Forward compatible with Drupal 12
- ✅ No deprecation warnings
- ✅ Tests run in isolated processes (better test isolation)
- ✅ All 17 tests still pass (8 Functional + 7 Kernel + 2 Unit)

## Testing
The CI/CD pipeline should now pass without deprecation warnings:
- Functional tests: 8 passed
- Kernel tests: 7 passed
- Unit tests: 2 passed
- **Total: 17/17 tests passing**

## References
- Drupal Change Record: https://www.drupal.org/node/3548485
- PHPUnit Documentation: https://docs.phpunit.de/en/11.5/attributes.html#runtestsinseparateprocesses
