<?php

namespace Drupal\body_class\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Link;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for displaying body class usage.
 */
class BodyClassListController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a BodyClassListController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(Connection $database, EntityTypeManagerInterface $entity_type_manager) {
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Displays a list of all nodes with body classes.
   *
   * @return array
   *   A render array.
   */
  public function listPage() {
    $header = [
      ['data' => $this->t('Node ID'), 'field' => 'bc.nid', 'sort' => 'desc'],
      ['data' => $this->t('Title')],
      ['data' => $this->t('Content Type')],
      ['data' => $this->t('CSS Class(es)')],
      ['data' => $this->t('Operations')],
    ];

    // Query the body_class table with table sort.
    $query = $this->database->select('body_class', 'bc')
      ->fields('bc', ['nid', 'css_class'])
      ->condition('css_class', '', '<>')
      ->extend('Drupal\Core\Database\Query\TableSortExtender')
      ->orderByHeader($header)
      ->extend('Drupal\Core\Database\Query\PagerSelectExtender')
      ->limit(50);

    $result = $query->execute();

    $rows = [];
    foreach ($result as $record) {
      $node = $this->entityTypeManager->getStorage('node')->load($record->nid);

      if ($node) {
        $rows[] = [
          $record->nid,
          Link::createFromRoute($node->label(), 'entity.node.canonical', ['node' => $node->id()]),
          $node->getType(),
          // Sanitize CSS class output to prevent XSS.
          ['data' => ['#plain_text' => $record->css_class]],
          Link::createFromRoute($this->t('Edit'), 'entity.node.edit_form', ['node' => $node->id()]),
        ];
      }
    }

    $build['description'] = [
      '#markup' => '<p>' . $this->t('This page shows all nodes that have custom body classes assigned.') . '</p>',
    ];

    if (!empty($rows)) {
      $build['table'] = [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No nodes with body classes found.'),
        '#attributes' => ['id' => 'body-class-list'],
      ];

      $build['pager'] = [
        '#type' => 'pager',
      ];
    }
    else {
      $build['empty'] = [
        '#markup' => '<p>' . $this->t('No nodes with body classes found.') . '</p>',
      ];
    }

    $build['back_link'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . Link::createFromRoute($this->t('← Back to settings'), 'body_class.settings')->toString() . '</p>',
    ];

    return $build;
  }

}
