<?php

namespace Drupal\body_class\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Body Class settings for this site.
 */
class BodyClassSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a BodyClassSettingsForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'body_class_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['body_class.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('body_class.settings');

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Configure which content types should have the Body Class field available.') . '</p>',
    ];

    // Get all content types.
    $node_types = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $content_type_options = [];
    foreach ($node_types as $type) {
      $content_type_options[$type->id()] = $type->label();
    }

    $enabled_types = $config->get('enabled_content_types') ?? [];

    // Check if '_all' is in the enabled types.
    $enable_all = in_array('_all', $enabled_types);

    $form['enable_all'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable for all content types'),
      '#description' => $this->t('If checked, the Body Class field will be available on all content types (including future ones).'),
      '#default_value' => $enable_all || empty($enabled_types),
    ];

    $form['enabled_content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled content types'),
      '#description' => $this->t('Select which content types should have the Body Class field. This is only used when "Enable for all content types" is unchecked.'),
      '#options' => $content_type_options,
      '#default_value' => !$enable_all ? $enabled_types : array_keys($content_type_options),
      '#states' => [
        'disabled' => [
          ':input[name="enable_all"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['view_usage'] = [
      '#type' => 'item',
      '#markup' => '<p>' . $this->t('View all nodes with body classes: <a href="@url">Body Class Usage List</a>', [
        '@url' => '/admin/config/development/body_class/list',
      ]) . '</p>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $enable_all = $form_state->getValue('enable_all');

    if ($enable_all) {
      // Store '_all' to indicate all content types are enabled.
      $enabled_types = ['_all'];
    }
    else {
      // Filter out unchecked values.
      $enabled_types = array_filter($form_state->getValue('enabled_content_types'));
      $enabled_types = array_values($enabled_types);
    }

    $this->config('body_class.settings')
      ->set('enabled_content_types', $enabled_types)
      ->save();

    // Clear cache so field definitions are rebuilt.
    drupal_flush_all_caches();

    parent::submitForm($form, $form_state);
  }

}
